#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
Stok Takip Uygulaması - İş Mantığı ve Veri Yönetimi
Türkçe tarih formatı düzeltmeleri eklendi
"""

import json
import os
from datetime import datetime, date
from typing import List, Dict, Any, Optional, Tuple
import copy

class StokManager:
    def __init__(self, data_file: str):
        """Stok yöneticisi sınıfı"""
        self.data_file = data_file
        self.islemler: List[Dict] = []
        self.stok_kartlari: List[Dict] = []
        self.settings: Dict = {
            'fontSize': 14
        }
        self.load_data()
    
    def load_data(self):
        """Veriyi dosyadan yükle"""
        try:
            if os.path.exists(self.data_file):
                with open(self.data_file, 'r', encoding='utf-8') as f:
                    data = json.load(f)
                    self.islemler = data.get('islemler', [])
                    self.stok_kartlari = data.get('stokKartlari', [])
                    self.settings = data.get('settings', {'fontSize': 14})
            else:
                # Boş verilerle başla
                self.islemler = []
                self.stok_kartlari = []
                self.settings = {'fontSize': 14}
        except Exception as e:
            print(f"Veri yükleme hatası: {e}")
            self.islemler = []
            self.stok_kartlari = []
            self.settings = {'fontSize': 14}
    
    def save_data(self):
        """Veriyi dosyaya kaydet"""
        try:
            data = {
                'islemler': self.islemler,
                'stokKartlari': self.stok_kartlari,
                'settings': self.settings,
                'lastUpdate': datetime.now().isoformat()
            }
            with open(self.data_file, 'w', encoding='utf-8') as f:
                json.dump(data, f, ensure_ascii=False, indent=2)
        except Exception as e:
            raise Exception(f"Veri kaydetme hatası: {e}")
    
    # FORMAT FONKSİYONLARI - TÜRKİYE STANDARTLARI
    def format_tarih(self, tarih_str: str, goster_yil: bool = True) -> str:
        """Tarihi Türkçe formatında göster (dd.mm.yyyy veya dd.mm)"""
        try:
            if not tarih_str:
                return ""
                
            # String değilse string'e çevir
            if not isinstance(tarih_str, str):
                tarih_str = str(tarih_str)
            
            tarih_dt = None
            
            # Çeşitli tarih formatlarını destekle
            if 'T' in tarih_str:  # ISO format (2024-12-01T10:30:00)
                tarih_dt = datetime.fromisoformat(tarih_str.replace('Z', '+00:00'))
            elif '-' in tarih_str and len(tarih_str) == 10:  # YYYY-MM-DD format
                tarih_dt = datetime.strptime(tarih_str, '%Y-%m-%d')
            elif '.' in tarih_str:  # DD.MM.YYYY format (Türkçe)
                if len(tarih_str) == 10:  # DD.MM.YYYY
                    tarih_dt = datetime.strptime(tarih_str, '%d.%m.%Y')
                elif len(tarih_str) == 5:  # DD.MM
                    tarih_dt = datetime.strptime(f"{tarih_str}.{datetime.now().year}", '%d.%m.%Y')
                else:
                    # Değişken uzunlukta format denemeleri
                    try:
                        tarih_dt = datetime.strptime(tarih_str, '%d.%m.%Y')
                    except:
                        # Başka formatlar dene
                        return tarih_str  # Parse edilemezse orijinal değeri döndür
            else:
                # Diğer durumlarda orijinal değeri döndür
                return tarih_str
            
            if tarih_dt is None:
                return tarih_str
            
            # Aynı yıl kontrolü - sadece aynı yılsa yılı gösterme
            current_year = datetime.now().year
            if not goster_yil or tarih_dt.year == current_year:
                return tarih_dt.strftime('%d.%m')  # Sadece gün.ay
            else:
                return tarih_dt.strftime('%d.%m.%Y')  # Tam Türkçe format: 01.12.2025
                
        except Exception as e:
            print(f"Tarih formatı hatası: {e}, input: {tarih_str}")
            return str(tarih_str)  # Hata durumunda orijinal değeri döndür
    
    def format_tarih_tam(self, tarih_str: str) -> str:
        """Tarihi tam Türkçe formatında göster (dd.mm.yyyy) - her zaman yıl ile"""
        try:
            if not tarih_str:
                return ""
                
            # String değilse string'e çevir
            if not isinstance(tarih_str, str):
                tarih_str = str(tarih_str)
            
            tarih_dt = None
            
            # Çeşitli tarih formatlarını destekle
            if 'T' in tarih_str:  # ISO format
                tarih_dt = datetime.fromisoformat(tarih_str.replace('Z', '+00:00'))
            elif '-' in tarih_str and len(tarih_str) == 10:  # YYYY-MM-DD format
                tarih_dt = datetime.strptime(tarih_str, '%Y-%m-%d')
            elif '.' in tarih_str:  # DD.MM.YYYY format
                if len(tarih_str) >= 10:  # DD.MM.YYYY
                    tarih_dt = datetime.strptime(tarih_str[:10], '%d.%m.%Y')
                elif len(tarih_str) == 5:  # DD.MM
                    tarih_dt = datetime.strptime(f"{tarih_str}.{datetime.now().year}", '%d.%m.%Y')
                else:
                    return tarih_str
            else:
                return tarih_str
            
            if tarih_dt is None:
                return tarih_str
                
            return tarih_dt.strftime('%d.%m.%Y')  # Her zaman tam format
                
        except Exception as e:
            print(f"Tam tarih formatı hatası: {e}, input: {tarih_str}")
            return str(tarih_str)
    
    def parse_turkish_date(self, tarih_str: str) -> str:
        """Türkçe tarih formatını (DD.MM.YYYY veya DD.MM) ISO formatına çevir"""
        try:
            if not tarih_str:
                return datetime.now().strftime('%Y-%m-%d')
                
            tarih_str = str(tarih_str).strip()
            
            # Zaten ISO format ise (YYYY-MM-DD)
            if '-' in tarih_str and len(tarih_str) == 10:
                try:
                    # Geçerli ISO format olup olmadığını kontrol et
                    datetime.strptime(tarih_str, '%Y-%m-%d')
                    return tarih_str
                except:
                    pass
            
            # DD.MM.YYYY formatı (10 karakter)
            if '.' in tarih_str and len(tarih_str) == 10:
                try:
                    dt = datetime.strptime(tarih_str, '%d.%m.%Y')
                    return dt.strftime('%Y-%m-%d')
                except:
                    pass
            
            # DD.MM formatı (5 karakter, mevcut yıl kullanılır)
            if '.' in tarih_str and len(tarih_str) == 5:
                try:
                    current_year = datetime.now().year
                    dt = datetime.strptime(f"{tarih_str}.{current_year}", '%d.%m.%Y')
                    return dt.strftime('%Y-%m-%d')
                except:
                    pass
            
            # D.M formatı (tek haneli gün/ay)
            if '.' in tarih_str and len(tarih_str) >= 3:
                try:
                    parts = tarih_str.split('.')
                    if len(parts) == 2:
                        day = int(parts[0])
                        month = int(parts[1])
                        current_year = datetime.now().year
                        dt = datetime(current_year, month, day)
                        return dt.strftime('%Y-%m-%d')
                except:
                    pass
            
            # Diğer durumlarda bugünün tarihini döndür
            return datetime.now().strftime('%Y-%m-%d')
                
        except Exception as e:
            print(f"Türkçe tarih parse hatası: {e}, input: {tarih_str}")
            # Hata durumunda bugünün tarihini döndür
            return datetime.now().strftime('%Y-%m-%d')
    
    def format_sayi(self, sayi: float) -> str:
        """Sayıyı Türkçe formatında göster (1.234,56)"""
        if sayi is None or sayi == '':
            return '0,00'
        try:
            return f"{float(sayi):,.2f}".replace(',', 'X').replace('.', ',').replace('X', '.')
        except:
            return '0,00'
    
    def format_sayi_basit(self, sayi: float) -> str:
        """Sayıyı basit formatta göster"""
        if sayi is None or sayi == '':
            return '0'
        try:
            if float(sayi) == int(sayi):
                return f"{int(sayi):,}".replace(',', '.')
            else:
                return f"{float(sayi):,.1f}".replace(',', 'X').replace('.', ',').replace('X', '.')
        except:
            return '0'
    
    # İŞLEM YÖNETİMİ
    def add_islem(self, islem_data: Dict) -> bool:
        """Yeni işlem ekle"""
        try:
            # Validasyon
            if not all(key in islem_data for key in ['islemTuru', 'tarih', 'faturaNo', 'urunIsim', 'miktar', 'fiyat']):
                raise ValueError("Eksik alan var!")
            
            # Tarih formatını normalize et (ISO formatına çevir)
            if isinstance(islem_data['tarih'], str) and '.' in islem_data['tarih']:
                islem_data['tarih'] = self.parse_turkish_date(islem_data['tarih'])
            
            # ÇIKIŞ işlemi için stok kontrolü
            if islem_data['islemTuru'] == 'ÇIKIŞ':
                mevcut_stok = self.hesapla_stok_tarihe_kadar(islem_data['urunIsim'], islem_data['tarih'])
                if islem_data['miktar'] > mevcut_stok:
                    raise ValueError(f"Stok yetersiz! Mevcut stok: {self.format_sayi_basit(mevcut_stok)}")
                if mevcut_stok <= 0:
                    raise ValueError("Bu ürün için stok bulunmuyor!")
            
            # Yeni işlem oluştur
            yeni_islem = {
                'id': self._generate_id(),
                'islemTuru': islem_data['islemTuru'],
                'tarih': islem_data['tarih'],  # ISO formatında sakla
                'faturaNo': islem_data['faturaNo'],
                'urunIsim': islem_data['urunIsim'],
                'miktar': float(islem_data['miktar']),
                'fiyat': float(islem_data['fiyat']),
                'birim': islem_data.get('birim', 'ADET'),
                'tutar': float(islem_data['miktar']) * float(islem_data['fiyat'])
            }
            
            self.islemler.append(yeni_islem)
            
            # Stok kartı kontrolü
            if not self.get_stok_karti_by_name(islem_data['urunIsim']):
                self.add_stok_karti_auto(islem_data['urunIsim'], yeni_islem['birim'])
            
            return True
        except Exception as e:
            raise e
    
    def update_islem(self, islem_id: int, islem_data: Dict) -> bool:
        """İşlemi güncelle"""
        try:
            islem_index = self._find_islem_index(islem_id)
            if islem_index == -1:
                raise ValueError("İşlem bulunamadı!")
            
            # Validasyon
            if not all(key in islem_data for key in ['islemTuru', 'tarih', 'faturaNo', 'urunIsim', 'miktar', 'fiyat']):
                raise ValueError("Eksik alan var!")
            
            # Tarih formatını normalize et
            if isinstance(islem_data['tarih'], str) and '.' in islem_data['tarih']:
                islem_data['tarih'] = self.parse_turkish_date(islem_data['tarih'])
            
            # Güncellenen işlem
            updated_islem = {
                'id': islem_id,
                'islemTuru': islem_data['islemTuru'],
                'tarih': islem_data['tarih'],
                'faturaNo': islem_data['faturaNo'],
                'urunIsim': islem_data['urunIsim'],
                'miktar': float(islem_data['miktar']),
                'fiyat': float(islem_data['fiyat']),
                'birim': islem_data.get('birim', 'ADET'),
                'tutar': float(islem_data['miktar']) * float(islem_data['fiyat'])
            }
            
            self.islemler[islem_index] = updated_islem
            return True
        except Exception as e:
            raise e
    
    def delete_islem(self, islem_id: int) -> bool:
        """İşlemi sil"""
        try:
            islem_index = self._find_islem_index(islem_id)
            if islem_index == -1:
                raise ValueError("İşlem bulunamadı!")
            
            del self.islemler[islem_index]
            return True
        except Exception as e:
            raise e
    
    def get_islem(self, islem_id: int) -> Optional[Dict]:
        """İşlemi getir"""
        for islem in self.islemler:
            if islem['id'] == islem_id:
                return islem
        return None
    
    def get_all_islemler(self, sort_by_date: bool = True) -> List[Dict]:
        """Tüm işlemleri getir"""
        if sort_by_date:
            return sorted(self.islemler, key=lambda x: x['tarih'], reverse=True)
        return self.islemler.copy()
    
    # STOK KARTI YÖNETİMİ
    def add_stok_karti(self, kdo: str, urun_ismi: str, birim: str) -> bool:
        """Yeni stok kartı ekle"""
        try:
            # Validasyon
            if not kdo or not urun_ismi:
                raise ValueError("KDO ve ürün ismi gerekli!")
            
            if self.get_stok_karti_by_kdo(kdo):
                raise ValueError("Bu stok kodu zaten mevcut!")
            
            if self.get_stok_karti_by_name(urun_ismi):
                raise ValueError("Bu ürün ismi zaten mevcut!")
            
            yeni_stok_karti = {
                'kdo': kdo,
                'urunIsmi': urun_ismi,
                'birim': birim
            }
            
            self.stok_kartlari.append(yeni_stok_karti)
            return True
        except Exception as e:
            raise e
    
    def add_stok_karti_auto(self, urun_ismi: str, birim: str = 'ADET') -> bool:
        """Otomatik KDO ile stok kartı ekle"""
        try:
            kdo = self.generate_auto_stok_kodu()
            return self.add_stok_karti(kdo, urun_ismi, birim)
        except Exception as e:
            raise e
    
    def update_stok_karti(self, old_kdo: str, new_kdo: str, urun_ismi: str, birim: str) -> bool:
        """Stok kartını güncelle"""
        try:
            stok_index = self._find_stok_karti_index(old_kdo)
            if stok_index == -1:
                raise ValueError("Stok kartı bulunamadı!")
            
            # KDO değiştirilmişse kontrol et
            if old_kdo != new_kdo and self.get_stok_karti_by_kdo(new_kdo):
                raise ValueError("Bu stok kodu zaten mevcut!")
            
            old_urun_ismi = self.stok_kartlari[stok_index]['urunIsmi']
            
            # Ürün ismi değiştirilmişse kontrol et
            if old_urun_ismi != urun_ismi and self.get_stok_karti_by_name(urun_ismi):
                raise ValueError("Bu ürün ismi zaten mevcut!")
            
            # Güncelle
            self.stok_kartlari[stok_index] = {
                'kdo': new_kdo,
                'urunIsmi': urun_ismi,
                'birim': birim
            }
            
            # İşlemlerdeki ürün ismini güncelle
            if old_urun_ismi != urun_ismi:
                for islem in self.islemler:
                    if islem['urunIsim'] == old_urun_ismi:
                        islem['urunIsim'] = urun_ismi
            
            return True
        except Exception as e:
            raise e
    
    def delete_stok_karti(self, kdo: str) -> bool:
        """Stok kartını sil"""
        try:
            stok_karti = self.get_stok_karti_by_kdo(kdo)
            if not stok_karti:
                raise ValueError("Stok kartı bulunamadı!")
            
            # İlgili işlemleri kontrol et
            ilgili_islemler = [i for i in self.islemler if i['urunIsim'] == stok_karti['urunIsmi']]
            if ilgili_islemler:
                raise ValueError(f"Bu stok kartı silinemez! {len(ilgili_islemler)} adet işlem bulunuyor.")
            
            stok_index = self._find_stok_karti_index(kdo)
            if stok_index != -1:
                del self.stok_kartlari[stok_index]
                return True
            
            return False
        except Exception as e:
            raise e
    
    def get_stok_karti_by_kdo(self, kdo: str) -> Optional[Dict]:
        """KDO ile stok kartı getir"""
        for stok in self.stok_kartlari:
            if stok['kdo'] == kdo:
                return stok
        return None
    
    def get_stok_karti_by_name(self, urun_ismi: str) -> Optional[Dict]:
        """Ürün ismi ile stok kartı getir"""
        for stok in self.stok_kartlari:
            if stok['urunIsmi'] == urun_ismi:
                return stok
        return None
    
    def get_all_stok_kartlari(self) -> List[Dict]:
        """Tüm stok kartlarını getir"""
        return self.stok_kartlari.copy()
    
    def generate_auto_stok_kodu(self) -> str:
        """Otomatik stok kodu üret"""
        stok_kodlari = [
            int(s['kdo'].replace('STOK-', '')) 
            for s in self.stok_kartlari 
            if s['kdo'].startswith('STOK-') and s['kdo'].replace('STOK-', '').isdigit()
        ]
        
        max_num = max(stok_kodlari) if stok_kodlari else 0
        next_num = max_num + 1
        return f"STOK-{next_num:03d}"
    
    def search_stok_kartlari(self, query: str) -> List[Dict]:
        """Stok kartlarında arama yap"""
        query = query.lower().strip()
        if not query:
            return []
        
        results = []
        for stok in self.stok_kartlari:
            if (query in stok['urunIsmi'].lower() or 
                query in stok['kdo'].lower()):
                results.append(stok)
        
        return results
    
    # STOK HESAPLAMALARI
    def hesapla_stok_tarihe_kadar(self, urun_ismi: str, tarih: str) -> float:
        """Belirli bir tarihe kadar olan stok miktarını hesapla"""
        try:
            # Tarih formatını normalize et
            if '.' in tarih:
                tarih = self.parse_turkish_date(tarih)
            
            tarih_dt = datetime.fromisoformat(tarih.replace('Z', '+00:00')).date()
        except:
            tarih_dt = datetime.now().date()
        
        stok = 0.0
        for islem in self.islemler:
            try:
                islem_tarih_str = islem['tarih']
                # Tarih formatını normalize et
                if '.' in islem_tarih_str:
                    islem_tarih_str = self.parse_turkish_date(islem_tarih_str)
                
                islem_tarih = datetime.fromisoformat(islem_tarih_str.replace('Z', '+00:00')).date()
            except:
                continue
            
            if islem['urunIsim'] == urun_ismi and islem_tarih <= tarih_dt:
                if islem['islemTuru'] == 'GİRİŞ':
                    stok += islem['miktar']
                elif islem['islemTuru'] == 'ÇIKIŞ':
                    stok -= islem['miktar']
        
        return stok
    
    def hesapla_stok_kartlari(self) -> List[Dict]:
        """Mevcut tüm stok kartlarının güncel durumunu hesapla"""
        hesaplamalar = {}
        
        # Stok kartları için başlangıç değerleri
        for kart in self.stok_kartlari:
            hesaplamalar[kart['urunIsmi']] = {
                **kart,
                'girisMiktari': 0.0,
                'cikisMiktari': 0.0,
                'kalanStok': 0.0,
                'girisTutari': 0.0,
                'cikisTutari': 0.0
            }
        
        # İşlemleri analiz et
        for islem in self.islemler:
            if islem['urunIsim'] not in hesaplamalar:
                # Yeni ürün
                yeni_kdo = f"URN-{len(hesaplamalar) + 1:03d}"
                hesaplamalar[islem['urunIsim']] = {
                    'kdo': yeni_kdo,
                    'urunIsmi': islem['urunIsim'],
                    'birim': islem.get('birim', 'ADET'),
                    'girisMiktari': 0.0,
                    'cikisMiktari': 0.0,
                    'kalanStok': 0.0,
                    'girisTutari': 0.0,
                    'cikisTutari': 0.0
                }
            
            stok = hesaplamalar[islem['urunIsim']]
            
            if islem['islemTuru'] == 'GİRİŞ':
                stok['girisMiktari'] += islem['miktar']
                stok['girisTutari'] += islem['tutar']
                stok['kalanStok'] += islem['miktar']
            elif islem['islemTuru'] == 'ÇIKIŞ':
                stok['cikisMiktari'] += islem['miktar']
                stok['cikisTutari'] += islem['tutar']
                stok['kalanStok'] -= islem['miktar']
        
        return list(hesaplamalar.values())
    
    def get_average_entry_price(self, urun_ismi: str) -> float:
        """Bir ürünün ortalama giriş fiyatını hesapla"""
        entry_operations = [i for i in self.islemler if i['urunIsim'] == urun_ismi and i['islemTuru'] == 'GİRİŞ']
        
        if not entry_operations:
            return 0.0
        
        total_value = sum(i['tutar'] for i in entry_operations)
        total_quantity = sum(i['miktar'] for i in entry_operations)
        
        return total_value / total_quantity if total_quantity > 0 else 0.0
    
    def get_fifo_stock_value(self, urun_ismi: str) -> float:
        """Bir ürünün FIFO yöntemiyle stok değerini hesapla"""
        urun_islemleri = [i for i in self.islemler if i['urunIsim'] == urun_ismi]
        urun_islemleri.sort(key=lambda x: (x['tarih'], x['id']))
        
        fifo_batches = []
        
        for islem in urun_islemleri:
            if islem['islemTuru'] == 'GİRİŞ':
                fifo_batches.append({
                    'tarih': islem['tarih'],
                    'miktar': islem['miktar'],
                    'fiyat': islem['fiyat']
                })
            elif islem['islemTuru'] == 'ÇIKIŞ':
                exit_quantity = islem['miktar']
                while exit_quantity > 0 and fifo_batches:
                    oldest_batch = fifo_batches[0]
                    consume_amount = min(exit_quantity, oldest_batch['miktar'])
                    
                    oldest_batch['miktar'] -= consume_amount
                    exit_quantity -= consume_amount
                    
                    if oldest_batch['miktar'] == 0:
                        fifo_batches.pop(0)
        
        # Kalan batches'ların toplam değeri
        return sum(batch['miktar'] * batch['fiyat'] for batch in fifo_batches)
    
    # İSTATİSTİKLER - FIFO DÜZELTMESİ ✅
    def get_statistics(self) -> Dict:
        """Uygulama istatistiklerini hesapla - FIFO yöntemi ile"""
        guncel_stok_kartlari = self.hesapla_stok_kartlari()
        
        total_stock_value = 0.0
        for stok in guncel_stok_kartlari:
            if stok['kalanStok'] > 0:
                # ✅ FIFO yöntemi ile stok değeri hesapla
                fifo_value = self.get_fifo_stock_value(stok['urunIsmi'])
                total_stock_value += fifo_value
        
        return {
            'totalProducts': len(guncel_stok_kartlari),
            'totalValue': total_stock_value,
            'totalEntries': len([i for i in self.islemler if i['islemTuru'] == 'GİRİŞ']),
            'totalExits': len([i for i in self.islemler if i['islemTuru'] == 'ÇIKIŞ'])
        }
    
    # URUN DETAY RAPORU
    def get_urun_detay_raporu(self, urun_ismi: str) -> Dict:
        """Ürün detay raporu (muavin dökümü)"""
        urun_islemleri = [i for i in self.islemler if i['urunIsim'] == urun_ismi]
        urun_islemleri.sort(key=lambda x: (x['tarih'], x['id']))
        
        detaylar = []
        kalan_stok = 0.0
        
        for islem in urun_islemleri:
            if islem['islemTuru'] == 'GİRİŞ':
                kalan_stok += islem['miktar']
            else:
                kalan_stok -= islem['miktar']
            
            detaylar.append({
                **islem,
                'kalanStok': kalan_stok
            })
        
        # Özet bilgiler
        toplam_giris = sum(i['miktar'] for i in urun_islemleri if i['islemTuru'] == 'GİRİŞ')
        toplam_cikis = sum(i['miktar'] for i in urun_islemleri if i['islemTuru'] == 'ÇIKIŞ')
        toplam_giris_tutar = sum(i['tutar'] for i in urun_islemleri if i['islemTuru'] == 'GİRİŞ')
        toplam_cikis_tutar = sum(i['tutar'] for i in urun_islemleri if i['islemTuru'] == 'ÇIKIŞ')
        
        return {
            'urunIsmi': urun_ismi,
            'detaylar': detaylar,
            'ozet': {
                'toplamGiris': toplam_giris,
                'toplamCikis': toplam_cikis,
                'kalanStok': kalan_stok,
                'toplamGirisTutar': toplam_giris_tutar,
                'toplamCikisTutar': toplam_cikis_tutar,
                'islemSayisi': len(urun_islemleri)
            }
        }
    
    # FIFO MALİYET HESAPLAMASI
    def calculate_fifo_report(self) -> List[Dict]:
        """FIFO maliyet raporu hesapla"""
        unique_products = list(set(i['urunIsim'] for i in self.islemler))
        fifo_report_data = []
        
        for urun_ismi in unique_products:
            urun_islemleri = [i for i in self.islemler if i['urunIsim'] == urun_ismi]
            urun_islemleri.sort(key=lambda x: (x['tarih'], x['id']))
            
            fifo_batches = []
            
            for islem in urun_islemleri:
                if islem['islemTuru'] == 'GİRİŞ':
                    fifo_batches.append({
                        'tarih': islem['tarih'],
                        'miktar': islem['miktar'],
                        'fiyat': islem['fiyat']
                    })
                elif islem['islemTuru'] == 'ÇIKIŞ':
                    exit_quantity = islem['miktar']
                    while exit_quantity > 0 and fifo_batches:
                        oldest_batch = fifo_batches[0]
                        consume_amount = min(exit_quantity, oldest_batch['miktar'])
                        
                        oldest_batch['miktar'] -= consume_amount
                        exit_quantity -= consume_amount
                        
                        if oldest_batch['miktar'] == 0:
                            fifo_batches.pop(0)
            
            kalan_miktar_fifo = sum(batch['miktar'] for batch in fifo_batches)
            alis_maliyeti_degeri = sum(batch['miktar'] * batch['fiyat'] for batch in fifo_batches)
            
            stok_karti = self.get_stok_karti_by_name(urun_ismi)
            birim = stok_karti['birim'] if stok_karti else 'ADET'
            
            fifo_report_data.append({
                'urunIsmi': urun_ismi,
                'kalanMiktarFIFO': kalan_miktar_fifo,
                'alisMaliyetiDegeri': alis_maliyeti_degeri,
                'birim': birim
            })
        
        return fifo_report_data
    
    # AYARLAR
    def save_settings(self, settings: Dict):
        """Ayarları kaydet"""
        self.settings.update(settings)
        self.save_data()
    
    def get_settings(self) -> Dict:
        """Ayarları getir"""
        return self.settings.copy()
    
    # KAR/ZARAR ANALİZİ
    def calculate_kar_zarar_analizi(self) -> List[Dict]:
        """Ürün bazında kar/zarar analizi hesapla"""
        unique_products = list(set(i['urunIsim'] for i in self.islemler))
        kar_zarar_data = []
        
        for urun_ismi in unique_products:
            urun_islemleri = [i for i in self.islemler if i['urunIsim'] == urun_ismi]
            
            # Giriş ve çıkış işlemlerini ayır
            giris_islemleri = [i for i in urun_islemleri if i['islemTuru'] == 'GİRİŞ']
            cikis_islemleri = [i for i in urun_islemleri if i['islemTuru'] == 'ÇIKIŞ']
            
            # Toplam alış ve satış hesapla
            toplam_alis_miktari = sum(i['miktar'] for i in giris_islemleri)
            toplam_alis_tutari = sum(i['tutar'] for i in giris_islemleri)
            
            toplam_satis_miktari = sum(i['miktar'] for i in cikis_islemleri)
            toplam_satis_tutari = sum(i['tutar'] for i in cikis_islemleri)
            
            # Ortalama alış ve satış fiyatları
            ort_alis_fiyati = toplam_alis_tutari / toplam_alis_miktari if toplam_alis_miktari > 0 else 0
            ort_satis_fiyati = toplam_satis_tutari / toplam_satis_miktari if toplam_satis_miktari > 0 else 0
            
            # Satılan miktarın alış maliyeti (FIFO benzeri)
            satilan_miktar_alis_maliyeti = self.calculate_sold_quantity_cost(urun_ismi)
            
            # Kar/Zarar hesaplama
            kar_zarar_tutari = toplam_satis_tutari - satilan_miktar_alis_maliyeti
            kar_zarar_yuzdesi = (kar_zarar_tutari / satilan_miktar_alis_maliyeti * 100) if satilan_miktar_alis_maliyeti > 0 else 0
            
            # Birim kar/zarar
            birim_kar_zarar = ort_satis_fiyati - ort_alis_fiyati
            
            # Stok kartı bilgisi
            stok_karti = self.get_stok_karti_by_name(urun_ismi)
            birim = stok_karti['birim'] if stok_karti else 'ADET'
            
            kar_zarar_data.append({
                'urunIsmi': urun_ismi,
                'birim': birim,
                'toplamAlisMiktari': toplam_alis_miktari,
                'toplamAlisTutari': toplam_alis_tutari,
                'toplamSatisMiktari': toplam_satis_miktari,
                'toplamSatisTutari': toplam_satis_tutari,
                'ortAlisFiyati': ort_alis_fiyati,
                'ortSatisFiyati': ort_satis_fiyati,
                'satilanMaliyeti': satilan_miktar_alis_maliyeti,
                'karZararTutari': kar_zarar_tutari,
                'karZararYuzdesi': kar_zarar_yuzdesi,
                'birimKarZarar': birim_kar_zarar
            })
        
        # Kar/zarar tutarına göre sırala (en karlıdan en zararlıya)
        kar_zarar_data.sort(key=lambda x: x['karZararTutari'], reverse=True)
        
        return kar_zarar_data
    
    def calculate_sold_quantity_cost(self, urun_ismi: str) -> float:
        """Satılan miktarın alış maliyetini FIFO yöntemiyle hesapla"""
        urun_islemleri = [i for i in self.islemler if i['urunIsim'] == urun_ismi]
        urun_islemleri.sort(key=lambda x: (x['tarih'], x['id']))
        
        # FIFO batch'leri
        fifo_batches = []
        total_sold_cost = 0.0
        
        for islem in urun_islemleri:
            if islem['islemTuru'] == 'GİRİŞ':
                fifo_batches.append({
                    'miktar': islem['miktar'],
                    'fiyat': islem['fiyat']
                })
            elif islem['islemTuru'] == 'ÇIKIŞ':
                exit_quantity = islem['miktar']
                while exit_quantity > 0 and fifo_batches:
                    oldest_batch = fifo_batches[0]
                    consume_amount = min(exit_quantity, oldest_batch['miktar'])
                    
                    # Bu miktar için alış maliyetini hesapla
                    cost = consume_amount * oldest_batch['fiyat']
                    total_sold_cost += cost
                    
                    oldest_batch['miktar'] -= consume_amount
                    exit_quantity -= consume_amount
                    
                    if oldest_batch['miktar'] == 0:
                        fifo_batches.pop(0)
        
        return total_sold_cost
    
    def get_kar_zarar_ozeti(self) -> Dict:
        """Genel kar/zarar özeti"""
        kar_zarar_data = self.calculate_kar_zarar_analizi()
        
        if not kar_zarar_data:
            return {
                'toplamKar': 0.0,
                'toplamZarar': 0.0,
                'netKarZarar': 0.0,
                'karliUrunSayisi': 0,
                'zararliUrunSayisi': 0,
                'toplamUrunSayisi': 0,
                'enKarliUrun': None,
                'enZararliUrun': None
            }
        
        toplam_kar = sum(item['karZararTutari'] for item in kar_zarar_data if item['karZararTutari'] > 0)
        toplam_zarar = sum(abs(item['karZararTutari']) for item in kar_zarar_data if item['karZararTutari'] < 0)
        net_kar_zarar = toplam_kar - toplam_zarar
        
        karli_urunler = [item for item in kar_zarar_data if item['karZararTutari'] > 0]
        zararli_urunler = [item for item in kar_zarar_data if item['karZararTutari'] < 0]
        
        en_karli = max(kar_zarar_data, key=lambda x: x['karZararTutari']) if kar_zarar_data else None
        en_zararli = min(kar_zarar_data, key=lambda x: x['karZararTutari']) if kar_zarar_data else None
        
        return {
            'toplamKar': toplam_kar,
            'toplamZarar': toplam_zarar,
            'netKarZarar': net_kar_zarar,
            'karliUrunSayisi': len(karli_urunler),
            'zararliUrunSayisi': len(zararli_urunler),
            'toplamUrunSayisi': len(kar_zarar_data),
            'enKarliUrun': en_karli,
            'enZararliUrun': en_zararli
        }
    
    def export_all_data(self) -> Dict:
        """Tüm veriyi dışa aktarma için hazırla"""
        return {
            'islemler': self.islemler,
            'stokKartlari': self.hesapla_stok_kartlari(),
            'tarih': datetime.now().isoformat()
        }
    
    def import_all_data(self, data: Dict) -> bool:
        """Veriyi içe aktar"""
        try:
            if 'islemler' in data and 'stokKartlari' in data:
                self.islemler = data['islemler']
                self.stok_kartlari = data['stokKartlari']
                return True
            return False
        except Exception:
            return False
    
    # YARDIMCI FONKSİYONLAR
    def _generate_id(self) -> int:
        """Benzersiz ID üret"""
        import time
        return int(time.time() * 1000)
    
    def _find_islem_index(self, islem_id: int) -> int:
        """İşlem indeksini bul"""
        for i, islem in enumerate(self.islemler):
            if islem['id'] == islem_id:
                return i
        return -1
    
    def _find_stok_karti_index(self, kdo: str) -> int:
        """Stok kartı indeksini bul"""
        for i, stok in enumerate(self.stok_kartlari):
            if stok['kdo'] == kdo:
                return i
        return -1