#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
Rapor Dışa Aktarma - Stok Takip Uygulaması
Türkçe tarih formatı ile güncellenmiş versiyon
"""

import csv
import os
from datetime import datetime
from typing import List, Dict

class ReportExporter:
    """Rapor dışa aktarma sınıfı"""
    
    def __init__(self, stok_manager):
        """Rapor exporter sınıfı"""
        self.stok_manager = stok_manager
    
    def format_sayi_excel(self, sayi: float) -> str:
        """Excel için sayı formatı (Türkçe: 1.234,56)"""
        if sayi is None or sayi == '':
            return '0,00'
        try:
            # Türkçe Excel formatı: binlik ayırıcı nokta, ondalık virgül
            formatted = f"{float(sayi):,.2f}"
            # İlk önce ondalık kısmı ayır
            parts = formatted.split('.')
            if len(parts) > 1:
                ondalik = parts[-1]
                binlik = '.'.join(parts[:-1])
                # Binlik ayırıcıları değiştir
                binlik = binlik.replace(',', '.')
                return f"{binlik},{ondalik}"
            else:
                return formatted.replace(',', '.') + ',00'
        except:
            return '0,00'
    
    def format_miktar_excel(self, sayi: float) -> str:
        """Excel için miktar formatı"""
        if sayi is None or sayi == '':
            return '0'
        try:
            if float(sayi) == int(sayi):
                return f"{int(sayi):,}".replace(',', '.')
            else:
                return f"{float(sayi):,.1f}".replace(',', 'X').replace('.', ',').replace('X', '.')
        except:
            return '0'
    
    def format_tarih_excel(self, tarih_str: str) -> str:
        """Excel için Türkçe tarih formatı (DD.MM.YYYY)"""
        return self.stok_manager.format_tarih_tam(tarih_str)
    
    def export_stok_kartlari_csv(self, filename: str):
        """Stok kartlarını CSV olarak dışa aktar"""
        guncel_stok_kartlari = self.stok_manager.hesapla_stok_kartlari()
        
        if not guncel_stok_kartlari:
            raise ValueError("Henüz stok kartı bulunmuyor!")
        
        with open(filename, 'w', newline='', encoding='utf-8-sig') as csvfile:
            writer = csv.writer(csvfile, delimiter=';')
            
            # Başlık satırı
            writer.writerow(['KDO', 'Ürün İsmi', 'Birim', 'Giriş', 'Çıkış', 'Kalan Stok', 'Giriş Tutarı', 'Çıkış Tutarı'])
            
            # Veri satırları
            for stok in guncel_stok_kartlari:
                row = [
                    stok['kdo'],
                    stok['urunIsmi'],
                    stok['birim'],
                    self.format_miktar_excel(stok.get('girisMiktari', 0)),
                    self.format_miktar_excel(stok.get('cikisMiktari', 0)),
                    self.format_miktar_excel(stok.get('kalanStok', 0)),
                    self.format_sayi_excel(stok.get('girisTutari', 0)),
                    self.format_sayi_excel(stok.get('cikisTutari', 0))
                ]
                writer.writerow(row)
    
    def export_islemler_csv(self, filename: str):
        """İşlemleri CSV olarak dışa aktar"""
        islemler = self.stok_manager.get_all_islemler(sort_by_date=False)
        
        if not islemler:
            raise ValueError("Henüz işlem bulunmuyor!")
        
        with open(filename, 'w', newline='', encoding='utf-8-sig') as csvfile:
            writer = csv.writer(csvfile, delimiter=';')
            
            # Başlık satırı
            writer.writerow(['İşlem Türü', 'Tarih', 'Fatura No', 'Ürün İsmi', 'Miktar', 'Birim', 'Fiyat', 'Tutar'])
            
            # Veri satırları
            for islem in islemler:
                row = [
                    islem['islemTuru'],
                    self.format_tarih_excel(islem['tarih']),  # Türkçe tarih formatı
                    islem['faturaNo'],
                    islem['urunIsim'],
                    self.format_miktar_excel(islem['miktar']),
                    islem.get('birim', 'ADET'),
                    self.format_sayi_excel(islem['fiyat']),
                    self.format_sayi_excel(islem['tutar'])
                ]
                writer.writerow(row)
    
    def export_urun_detay_csv(self, filename: str, urun_ismi: str):
        """Ürün detay raporunu CSV olarak dışa aktar - Düzeltildi"""
        # Fonksiyon kendi içinde veriyi alıyor
        rapor = self.stok_manager.get_urun_detay_raporu(urun_ismi)
        
        with open(filename, 'w', newline='', encoding='utf-8-sig') as csvfile:
            writer = csv.writer(csvfile, delimiter=';')
            
            # Başlık ve boş satır
            writer.writerow([f'{urun_ismi} - Muavin Dökümü'])
            writer.writerow([])
            
            # Tablo başlığı
            writer.writerow(['Tarih', 'İşlem Türü', 'Fatura No', 'Miktar', 'Birim', 'Fiyat', 'Tutar', 'Kalan Stok', 'Kalan Stok Birim'])
            
            # Detay verileri
            for detay in rapor['detaylar']:
                row = [
                    self.format_tarih_excel(detay['tarih']),  # Türkçe tarih formatı
                    detay['islemTuru'],
                    detay['faturaNo'],
                    self.format_miktar_excel(detay['miktar']),
                    detay.get('birim', 'ADET'),
                    self.format_sayi_excel(detay['fiyat']),
                    self.format_sayi_excel(detay['tutar']),
                    self.format_miktar_excel(detay['kalanStok']),
                    detay.get('birim', 'ADET')
                ]
                writer.writerow(row)
            
            # Özet bilgiler
            writer.writerow([])
            writer.writerow(['ÖZET BİLGİLER', '', ''])
            
            ozet = rapor['ozet']
            birim = rapor['detaylar'][0].get('birim', 'ADET') if rapor['detaylar'] else 'ADET'
            
            ozet_rows = [
                ['Toplam Giriş', self.format_miktar_excel(ozet['toplamGiris']), birim],
                ['Toplam Çıkış', self.format_miktar_excel(ozet['toplamCikis']), birim],
                ['Kalan Stok', self.format_miktar_excel(ozet['kalanStok']), birim],
                ['Giriş Tutarı', self.format_sayi_excel(ozet['toplamGirisTutar']), ''],
                ['Çıkış Tutarı', self.format_sayi_excel(ozet['toplamCikisTutar']), ''],
                ['İşlem Sayısı', str(ozet['islemSayisi']), '']
            ]
            
            for row in ozet_rows:
                writer.writerow(row)
    
    def export_fifo_report_csv(self, filename: str, fifo_data: List[Dict]):
        """FIFO maliyet raporunu CSV olarak dışa aktar"""
        if not fifo_data:
            raise ValueError("Henüz FIFO raporu için veri bulunmuyor!")
        
        with open(filename, 'w', newline='', encoding='utf-8-sig') as csvfile:
            writer = csv.writer(csvfile, delimiter=';')
            
            # Başlık
            writer.writerow(['FIFO Maliyet Raporu'])
            writer.writerow([])
            writer.writerow([f'Rapor Tarihi: {self.format_tarih_excel(datetime.now().isoformat())}'])
            writer.writerow([])
            
            # Tablo başlığı
            writer.writerow(['Ürün İsmi', 'Kalan Miktar (FIFO)', 'Birim', 'Alış Maliyeti Değeri (FIFO)'])
            
            total_fifo_value = 0.0
            
            # Veri satırları
            for item in fifo_data:
                total_fifo_value += item['alisMaliyetiDegeri']
                row = [
                    item['urunIsmi'],
                    self.format_miktar_excel(item['kalanMiktarFIFO']),
                    item['birim'],
                    self.format_sayi_excel(item['alisMaliyetiDegeri'])
                ]
                writer.writerow(row)
            
            # Toplam satır
            writer.writerow([])
            writer.writerow(['GENEL TOPLAM', '', '', self.format_sayi_excel(total_fifo_value)])
    
    def export_kar_zarar_csv(self, filename: str):
        """Kar/Zarar analizini CSV olarak dışa aktar"""
        kar_zarar_data = self.stok_manager.calculate_kar_zarar_analizi()
        
        if not kar_zarar_data:
            raise ValueError("Henüz kar/zarar analizi için veri bulunmuyor!")
        
        with open(filename, 'w', newline='', encoding='utf-8-sig') as csvfile:
            writer = csv.writer(csvfile, delimiter=';')
            
            # Başlık
            writer.writerow(['KAR/ZARAR ANALİZİ'])
            writer.writerow([])
            writer.writerow([f'Rapor Tarihi: {self.format_tarih_excel(datetime.now().isoformat())}'])
            writer.writerow([])
            
            # Özet bilgiler
            ozet = self.stok_manager.get_kar_zarar_ozeti()
            writer.writerow(['ÖZET BİLGİLER'])
            writer.writerow(['Net Kar/Zarar', self.format_sayi_excel(ozet['netKarZarar'])])
            writer.writerow(['Toplam Kar', self.format_sayi_excel(ozet['toplamKar'])])
            writer.writerow(['Toplam Zarar', self.format_sayi_excel(ozet['toplamZarar'])])
            writer.writerow(['Karlı Ürün Sayısı', str(ozet['karliUrunSayisi'])])
            writer.writerow(['Zararlı Ürün Sayısı', str(ozet['zararliUrunSayisi'])])
            
            if ozet.get('enKarliUrun'):
                writer.writerow(['En Karlı Ürün', 
                               f"{ozet['enKarliUrun']['urunIsmi']} ({self.format_sayi_excel(ozet['enKarliUrun']['karZararTutari'])} TL)"])
            
            if ozet.get('enZararliUrun'):
                writer.writerow(['En Zararlı Ürün', 
                               f"{ozet['enZararliUrun']['urunIsmi']} ({self.format_sayi_excel(ozet['enZararliUrun']['karZararTutari'])} TL)"])
            
            writer.writerow([])
            
            # Tablo başlığı
            writer.writerow(['Ürün İsmi', 'Birim', 'Alış Miktarı', 'Satış Miktarı', 
                           'Ortalama Alış Fiyatı', 'Ortalama Satış Fiyatı', 
                           'Kar/Zarar Tutarı', 'Kar/Zarar Yüzdesi', 'Birim Kar/Zarar'])
            
            # Veri satırları
            for item in kar_zarar_data:
                row = [
                    item['urunIsmi'],
                    item['birim'],
                    self.format_miktar_excel(item['toplamAlisMiktari']),
                    self.format_miktar_excel(item['toplamSatisMiktari']),
                    self.format_sayi_excel(item['ortAlisFiyati']),
                    self.format_sayi_excel(item['ortSatisFiyati']),
                    self.format_sayi_excel(item['karZararTutari']),
                    f"{item['karZararYuzdesi']:.2f}%".replace('.', ','),
                    self.format_sayi_excel(item['birimKarZarar'])
                ]
                writer.writerow(row)
            
            # Genel toplam
            toplam_kar_zarar = sum(item['karZararTutari'] for item in kar_zarar_data)
            writer.writerow([])
            writer.writerow(['GENEL TOPLAM', '', '', '', '', '', 
                           self.format_sayi_excel(toplam_kar_zarar), '', ''])