#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
Ana Pencere - Stok Takip Uygulaması
Türkçe tarih formatı ile güncellenmiş versiyon
"""

import tkinter as tk
from tkinter import ttk, messagebox, filedialog
import json
import os
from datetime import datetime
from .dialogs import IslemDialog, StokKartiDialog, UrunDetayDialog, FifoRaporDialog
from .reports import ReportExporter

class MainWindow:
    def __init__(self, root, stok_manager):
        """Ana pencere sınıfı"""
        self.root = root
        self.stok_manager = stok_manager
        self.current_font_size = stok_manager.get_settings().get('fontSize', 14)
        
        # Stil ayarları
        self.setup_styles()
        
        # Ana frame
        self.main_frame = ttk.Frame(root)
        self.main_frame.pack(fill=tk.BOTH, expand=True, padx=10, pady=10)
        
        # GUI bileşenlerini oluştur
        self.create_header()
        self.create_stats()
        self.create_import_export_buttons()
        self.create_calculator()  # Hesap makinesi ekle
        self.create_tabs()
        self.create_status_bar()
        
        # Veriyi yükle ve göster
        self.refresh_all()
        
        # Klavye kısayolları
        self.root.bind('<Control-n>', lambda e: self.open_islem_dialog())
        self.root.bind('<Control-s>', lambda e: self.stok_manager.save_data())
        self.root.bind('<F5>', lambda e: self.refresh_all())
        self.root.bind('<Control-Tab>', lambda e: self.switch_tab())
        self.root.bind('<Control-Right>', lambda e: self.switch_tab())
        self.root.bind('<Control-Left>', lambda e: self.switch_tab())
        self.root.bind('<Alt-Right>', lambda e: self.switch_tab())
        self.root.bind('<Alt-Left>', lambda e: self.switch_tab())
    
    def setup_styles(self):
        """GUI stillerini ayarla"""
        style = ttk.Style()
        
        # Başlık renkleri - lacivert ve koyu
        style.configure("Header.TLabel", 
                       foreground="#000080",  # Lacivert
                       background="#ffffff",   # Beyaz arka plan
                       font=('Arial', 18, 'bold'))
        
        style.configure("Stat.TLabel", 
                       foreground="#1a1a1a",  # Çok koyu gri (neredeyse siyah)
                       background="#ffffff",   # Beyaz arka plan
                       font=('Arial', 16, 'bold'))
        
        # Treeview başlıkları - lacivert
        style.configure("Treeview.Heading", 
                       background="#e5e7eb",  # Açık gri arka plan
                       foreground="#000080",  # Lacivert
                       font=('Arial', self.current_font_size, 'bold'),
                       relief="raised",
                       borderwidth=1)
        
        # LabelFrame başlıkları - lacivert ve koyu
        style.configure("TLabelframe.Label", 
                       foreground="#000080",  # Lacivert
                       background="#ffffff",   # Beyaz arka plan
                       font=('Arial', self.current_font_size, 'bold'))
        
        # Button renkler
        style.configure("Success.TButton", 
                       foreground="#000080",  # Lacivert yazı
                       background="#e6f7e6",  # Açık yeşil arka plan
                       font=('Arial', self.current_font_size, 'bold'))
        style.map("Success.TButton",
                 background=[('active', '#d4edda'), ('!active', '#e6f7e6')],
                 foreground=[('active', '#000080'), ('!active', '#000080')])
        
        style.configure("Primary.TButton", 
                       foreground="#000080",  # Lacivert yazı
                       background="#e6f0ff",  # Açık mavi arka plan
                       font=('Arial', self.current_font_size, 'bold'))
        style.map("Primary.TButton",
                 background=[('active', '#cce0ff'), ('!active', '#e6f0ff')],
                 foreground=[('active', '#000080'), ('!active', '#000080')])
        
        # Normal butonlar
        style.configure("TButton",
                       font=('Arial', self.current_font_size))
        
        # Label'lar - koyu gri
        style.configure("TLabel",
                       font=('Arial', self.current_font_size),
                       foreground="#1a1a1a")  # Koyu gri
    
    def create_header(self):
        """Başlık bölümü"""
        header_frame = ttk.Frame(self.main_frame)
        header_frame.pack(fill=tk.X, pady=(0, 10))
        
        # Başlık
        title_label = ttk.Label(header_frame, text="📦 Stok Takip Uygulaması", 
                               style="Header.TLabel")
        title_label.pack(side=tk.LEFT)
        
        # Sağ taraf - font boyutu ve yeni işlem butonu
        right_frame = ttk.Frame(header_frame)
        right_frame.pack(side=tk.RIGHT)
        
        # Font boyutu kontrolleri
        font_frame = ttk.Frame(right_frame)
        font_frame.pack(side=tk.LEFT, padx=(0, 20))
        
        ttk.Label(font_frame, text="Yazı Boyutu:").pack(side=tk.LEFT)
        ttk.Button(font_frame, text="-", width=3, 
                  command=lambda: self.change_font_size(-1)).pack(side=tk.LEFT, padx=2)
        self.font_size_label = ttk.Label(font_frame, text=f"{self.current_font_size}px")
        self.font_size_label.pack(side=tk.LEFT, padx=5)
        ttk.Button(font_frame, text="+", width=3, 
                  command=lambda: self.change_font_size(1)).pack(side=tk.LEFT, padx=2)
        
        # Yeni işlem butonu
        ttk.Button(right_frame, text="➕ Yeni İşlem Ekle", 
                  style="Success.TButton",
                  command=self.open_islem_dialog).pack(side=tk.LEFT)
    
    def create_stats(self):
        """İstatistik kartları"""
        stats_frame = ttk.Frame(self.main_frame)
        stats_frame.pack(fill=tk.X, pady=(0, 10))
        
        # 4 adet istatistik kartı
        self.stats_vars = {
            'totalProducts': tk.StringVar(value="0"),
            'totalValue': tk.StringVar(value="0,00 TL"),
            'totalEntries': tk.StringVar(value="0"),
            'totalExits': tk.StringVar(value="0")
        }
        
        stats_labels = [
            ("Toplam Ürün Çeşidi", "totalProducts"),
            ("Stok Değeri (FIFO)", "totalValue"),
            ("Toplam Giriş İşlemi", "totalEntries"),
            ("Toplam Çıkış İşlemi", "totalExits")
        ]
        
        for i, (label_text, var_key) in enumerate(stats_labels):
            card_frame = ttk.LabelFrame(stats_frame, text=label_text, padding=10)
            card_frame.grid(row=0, column=i, padx=5, sticky="ew")
            stats_frame.columnconfigure(i, weight=1)
            
            value_label = ttk.Label(card_frame, textvariable=self.stats_vars[var_key], 
                                   style="Stat.TLabel")
            value_label.pack()
    
    def create_import_export_buttons(self):
        """İçe/Dışa aktarma butonları"""
        import_export_frame = ttk.Frame(self.main_frame)
        import_export_frame.pack(fill=tk.X, pady=(0, 10))
        
        buttons = [
            ("⬆️ Veri İçe Aktar", self.import_data, "Primary.TButton"),
            ("⬇️ Tüm Veriyi Dışa Aktar", self.export_data, "Success.TButton"),
            ("⬇️ Stok Kartları Excel", self.export_stok_kartlari_excel, "Success.TButton"),
            ("⬇️ İşlemler Detay Excel", self.export_islemler_excel, "Success.TButton")
        ]
        
        for i, (text, command, style) in enumerate(buttons):
            btn = ttk.Button(import_export_frame, text=text, command=command, style=style)
            btn.grid(row=0, column=i, padx=5, sticky="ew")
            import_export_frame.columnconfigure(i, weight=1)
    
    def create_calculator(self):
        """Hesap makinesi oluştur"""
        calc_frame = ttk.LabelFrame(self.main_frame, text="🧮 Hesap Makinesi", padding=5)
        calc_frame.pack(fill=tk.X, pady=(0, 10))
        
        # Ana layout - yatay
        main_calc_frame = ttk.Frame(calc_frame)
        main_calc_frame.pack(fill=tk.X)
        
        # Sol taraf - display ve temel butonlar
        left_frame = ttk.Frame(main_calc_frame)
        left_frame.pack(side=tk.LEFT, fill=tk.X, expand=True)
        
        # Hesap makinesi değişkenleri
        self.calc_display = tk.StringVar(value="0")
        self.calc_memory = 0
        self.calc_operation = None
        self.calc_waiting_operand = False
        
        # Display
        display_frame = ttk.Frame(left_frame)
        display_frame.pack(fill=tk.X, pady=(0, 5))
        
        display_entry = ttk.Entry(display_frame, textvariable=self.calc_display, 
                                 font=('Arial', 14, 'bold'), justify='right', 
                                 state='readonly', width=20)
        display_entry.pack(side=tk.LEFT, fill=tk.X, expand=True)
        
        # Temizle butonu
        ttk.Button(display_frame, text="C", width=4, 
                  command=self.calc_clear).pack(side=tk.LEFT, padx=(5, 0))
        
        # Sağ taraf - kompakt buton grid
        right_frame = ttk.Frame(main_calc_frame)
        right_frame.pack(side=tk.RIGHT, padx=(10, 0))
        
        # Kompakt buton grid (3x4)
        buttons = [
            ['7', '8', '9', '/'],
            ['4', '5', '6', '*'],
            ['1', '2', '3', '-'],
            ['0', '.', '=', '+']
        ]
        
        for row, button_row in enumerate(buttons):
            for col, text in enumerate(button_row):
                if text == '=':
                    btn = ttk.Button(right_frame, text=text, width=4,
                                   command=self.calc_equals, style="Success.TButton")
                elif text in ['+', '-', '*', '/']:
                    btn = ttk.Button(right_frame, text=text, width=4,
                                   command=lambda t=text: self.calc_operation_click(t),
                                   style="Primary.TButton")
                elif text == '.':
                    btn = ttk.Button(right_frame, text=text, width=4,
                                   command=self.calc_decimal)
                else:
                    btn = ttk.Button(right_frame, text=text, width=4,
                                   command=lambda t=text: self.calc_number_click(t))
                
                btn.grid(row=row, column=col, padx=1, pady=1, sticky="ew")
        
        # Klavye bağlantıları - ana pencereye bağla
        self.setup_calculator_keyboard()
    
    def setup_calculator_keyboard(self):
        """Hesap makinesi klavye kısayolları"""
        # Numpad ve normal sayı tuşları
        for i in range(10):
            self.root.bind(f'<Key-{i}>', lambda e, num=str(i): self.calc_number_click(num))
            self.root.bind(f'<KP_{i}>', lambda e, num=str(i): self.calc_number_click(num))
        
        # Operatörler
        self.root.bind('<plus>', lambda e: self.calc_operation_click('+'))
        self.root.bind('<KP_Add>', lambda e: self.calc_operation_click('+'))
        self.root.bind('<minus>', lambda e: self.calc_operation_click('-'))
        self.root.bind('<KP_Subtract>', lambda e: self.calc_operation_click('-'))
        self.root.bind('<asterisk>', lambda e: self.calc_operation_click('*'))
        self.root.bind('<KP_Multiply>', lambda e: self.calc_operation_click('*'))
        self.root.bind('<slash>', lambda e: self.calc_operation_click('/'))
        self.root.bind('<KP_Divide>', lambda e: self.calc_operation_click('/'))
        
        # Enter ve = tuşları
        self.root.bind('<Return>', lambda e: self.calc_equals())
        self.root.bind('<KP_Enter>', lambda e: self.calc_equals())
        self.root.bind('<equal>', lambda e: self.calc_equals())
        
        # Ondalık nokta
        self.root.bind('<period>', lambda e: self.calc_decimal())
        self.root.bind('<KP_Decimal>', lambda e: self.calc_decimal())
        
        # Temizle
        self.root.bind('<Delete>', lambda e: self.calc_clear())
        self.root.bind('<BackSpace>', lambda e: self.calc_backspace())
        
        # Escape
        self.root.bind('<Escape>', lambda e: self.calc_clear())
    
    def calc_number_click(self, number):
        """Sayı butonuna tıklanma"""
        if self.calc_waiting_operand:
            self.calc_display.set(number)
            self.calc_waiting_operand = False
        else:
            current = self.calc_display.get()
            if current == "0":
                self.calc_display.set(number)
            else:
                self.calc_display.set(current + number)
    
    def calc_operation_click(self, operation):
        """Operatör butonuna tıklanma"""
        try:
            # Türkçe formatı (virgül) noktaya çevir
            current_str = self.calc_display.get().replace(',', '.')
            current_value = float(current_str)
            
            if self.calc_operation is not None:
                self.calc_equals()
                current_str = self.calc_display.get().replace(',', '.')
                current_value = float(current_str)
            
            self.calc_memory = current_value
            self.calc_operation = operation
            self.calc_waiting_operand = True
        except ValueError:
            self.calc_display.set("Hata")
    
    def calc_equals(self):
        """Eşittir işlemi"""
        try:
            if self.calc_operation is not None:
                # Türkçe formatı (virgül) noktaya çevir
                current_str = self.calc_display.get().replace(',', '.')
                current_value = float(current_str)
                
                if self.calc_operation == '+':
                    result = self.calc_memory + current_value
                elif self.calc_operation == '-':
                    result = self.calc_memory - current_value
                elif self.calc_operation == '*':
                    result = self.calc_memory * current_value
                elif self.calc_operation == '/':
                    if current_value == 0:
                        self.calc_display.set("Sıfıra bölme hatası")
                        return
                    result = self.calc_memory / current_value
                
                # Sonucu formatla (Türkçe sayı formatı)
                if result == int(result):
                    formatted_result = str(int(result))
                else:
                    formatted_result = f"{result:.2f}".replace('.', ',')
                
                self.calc_display.set(formatted_result)
                self.calc_operation = None
                self.calc_waiting_operand = True
        except (ValueError, ZeroDivisionError):
            self.calc_display.set("Hata")
    
    def calc_decimal(self):
        """Ondalık nokta"""
        current = self.calc_display.get()
        if self.calc_waiting_operand:
            self.calc_display.set("0,")
            self.calc_waiting_operand = False
        elif ',' not in current:
            self.calc_display.set(current + ',')
    
    def calc_clear(self):
        """Temizle"""
        self.calc_display.set("0")
        self.calc_memory = 0
        self.calc_operation = None
        self.calc_waiting_operand = False
    
    def calc_backspace(self):
        """Geri tuşu"""
        current = self.calc_display.get()
        if len(current) > 1:
            self.calc_display.set(current[:-1])
        else:
            self.calc_display.set("0")
    
    def create_tabs(self):
        """Tab sistemi"""
        # Notebook (Tab Container)
        self.notebook = ttk.Notebook(self.main_frame)
        self.notebook.pack(fill=tk.BOTH, expand=True)
        
        # İşlemler tabı
        self.islemler_frame = ttk.Frame(self.notebook)
        self.notebook.add(self.islemler_frame, text="İşlemler")
        self.create_islemler_tab()
        
        # Stok kartları tabı
        self.stok_frame = ttk.Frame(self.notebook)
        self.notebook.add(self.stok_frame, text="Stok Kartları")
        self.create_stok_tab()
        
        # Kar/Zarar Analizi tabı
        self.kar_zarar_frame = ttk.Frame(self.notebook)
        self.notebook.add(self.kar_zarar_frame, text="💰 Kar/Zarar Analizi")
        self.create_kar_zarar_tab()
        
        # Tab değişim eventi
        self.notebook.bind('<<NotebookTabChanged>>', self.on_tab_changed)
    
    def create_islemler_tab(self):
        """İşlemler tabı"""
        # Treeview için frame
        tree_frame = ttk.Frame(self.islemler_frame)
        tree_frame.pack(fill=tk.BOTH, expand=True, padx=5, pady=5)
        
        # Scrollbar
        tree_scroll = ttk.Scrollbar(tree_frame)
        tree_scroll.pack(side=tk.RIGHT, fill=tk.Y)
        
        # Treeview
        columns = ("Tür", "Tarih", "Fatura No", "Ürün İsmi", "Miktar", "Fiyat", "Tutar")
        self.islemler_tree = ttk.Treeview(tree_frame, columns=columns, show='headings', 
                                         yscrollcommand=tree_scroll.set)
        tree_scroll.config(command=self.islemler_tree.yview)
        
        # Sütun başlıkları ve genişlikleri
        column_widths = {"Tür": 80, "Tarih": 90, "Fatura No": 120, "Ürün İsmi": 200, 
                        "Miktar": 100, "Fiyat": 100, "Tutar": 100}
        
        for col in columns:
            self.islemler_tree.heading(col, text=col, command=lambda c=col: self.sort_islemler(c))
            self.islemler_tree.column(col, width=column_widths.get(col, 100), anchor='center')
        
        # Ürün İsmi sütunu sola hizalı
        self.islemler_tree.column("Ürün İsmi", anchor='w')
        
        self.islemler_tree.pack(fill=tk.BOTH, expand=True)
        
        # Sağ tık menüsü
        self.create_islemler_context_menu()
        
        # Çift tık eventi
        self.islemler_tree.bind('<Double-1>', self.on_islem_double_click)
    
    def create_stok_tab(self):
        """Stok kartları tabı"""
        # Üst butonlar
        button_frame = ttk.Frame(self.stok_frame)
        button_frame.pack(fill=tk.X, padx=5, pady=5)
        
        ttk.Button(button_frame, text="➕ Yeni Stok Kartı", 
                  style="Success.TButton",
                  command=self.open_stok_karti_dialog).pack(side=tk.LEFT, padx=(0, 10))
        ttk.Button(button_frame, text="📊 FIFO Maliyet Raporu", 
                  style="Primary.TButton",
                  command=self.open_fifo_rapor_dialog).pack(side=tk.LEFT)
        
        # Treeview için frame
        tree_frame = ttk.Frame(self.stok_frame)
        tree_frame.pack(fill=tk.BOTH, expand=True, padx=5, pady=5)
        
        # Scrollbar
        tree_scroll = ttk.Scrollbar(tree_frame)
        tree_scroll.pack(side=tk.RIGHT, fill=tk.Y)
        
        # Treeview
        columns = ("KDO", "Ürün İsmi", "Birim", "Giriş", "Çıkış", "Kalan Stok", "Giriş Tutarı", "Çıkış Tutarı")
        self.stok_tree = ttk.Treeview(tree_frame, columns=columns, show='headings', 
                                     yscrollcommand=tree_scroll.set)
        tree_scroll.config(command=self.stok_tree.yview)
        
        # Sütun başlıkları ve genişlikleri
        column_widths = {"KDO": 100, "Ürün İsmi": 200, "Birim": 60, "Giriş": 80, 
                        "Çıkış": 80, "Kalan Stok": 90, "Giriş Tutarı": 110, "Çıkış Tutarı": 110}
        
        for col in columns:
            self.stok_tree.heading(col, text=col, command=lambda c=col: self.sort_stok(c))
            self.stok_tree.column(col, width=column_widths.get(col, 100), anchor='center')
        
        # Ürün İsmi sütunu sola hizalı
        self.stok_tree.column("Ürün İsmi", anchor='w')
        
        self.stok_tree.pack(fill=tk.BOTH, expand=True)
        
        # Sağ tık menüsü
        self.create_stok_context_menu()
        
    def create_kar_zarar_tab(self):
        """Kar/Zarar analizi tabı"""
        # Özet kartlar
        ozet_frame = ttk.Frame(self.kar_zarar_frame)
        ozet_frame.pack(fill=tk.X, padx=5, pady=5)
        
        # Özet istatistik değişkenleri
        self.kar_zarar_vars = {
            'netKarZarar': tk.StringVar(value="0,00 TL"),
            'toplamKar': tk.StringVar(value="0,00 TL"),
            'toplamZarar': tk.StringVar(value="0,00 TL"),
            'karliUrunSayisi': tk.StringVar(value="0"),
            'zararliUrunSayisi': tk.StringVar(value="0"),
            'enKarliUrun': tk.StringVar(value="-"),
            'enZararliUrun': tk.StringVar(value="-")
        }
        
        # Üst satır - ana istatistikler
        ust_stats = [
            ("Net Kar/Zarar", "netKarZarar"),
            ("Toplam Kar", "toplamKar"),
            ("Toplam Zarar", "toplamZarar")
        ]
        
        for i, (label_text, var_key) in enumerate(ust_stats):
            card_frame = ttk.LabelFrame(ozet_frame, text=label_text, padding=10)
            card_frame.grid(row=0, column=i, padx=5, sticky="ew")
            ozet_frame.columnconfigure(i, weight=1)
            
            value_label = ttk.Label(card_frame, textvariable=self.kar_zarar_vars[var_key], 
                                   style="Stat.TLabel")
            value_label.pack()
        
        # Alt satır - ürün istatistikleri
        alt_stats = [
            ("Karlı Ürün Sayısı", "karliUrunSayisi"),
            ("Zararlı Ürün Sayısı", "zararliUrunSayisi")
        ]
        
        for i, (label_text, var_key) in enumerate(alt_stats):
            card_frame = ttk.LabelFrame(ozet_frame, text=label_text, padding=10)
            card_frame.grid(row=1, column=i, padx=5, pady=(10, 0), sticky="ew")
            
            value_label = ttk.Label(card_frame, textvariable=self.kar_zarar_vars[var_key], 
                                   style="Stat.TLabel")
            value_label.pack()
        
        # En karlı/zararlı ürün kartı
        extreme_frame = ttk.LabelFrame(ozet_frame, text="Ekstrem Durumlar", padding=10)
        extreme_frame.grid(row=1, column=2, padx=5, pady=(10, 0), sticky="ew")
        
        ttk.Label(extreme_frame, text="En Karlı:", font=('Arial', 10, 'bold')).pack(anchor='w')
        ttk.Label(extreme_frame, textvariable=self.kar_zarar_vars['enKarliUrun']).pack(anchor='w')
        
        ttk.Label(extreme_frame, text="En Zararlı:", font=('Arial', 10, 'bold')).pack(anchor='w', pady=(5, 0))
        ttk.Label(extreme_frame, textvariable=self.kar_zarar_vars['enZararliUrun']).pack(anchor='w')
        
        # Yenile butonu
        button_frame = ttk.Frame(self.kar_zarar_frame)
        button_frame.pack(fill=tk.X, padx=5, pady=5)
        
        ttk.Button(button_frame, text="🔄 Analizi Yenile", 
                  style="Primary.TButton",
                  command=self.refresh_kar_zarar).pack(side=tk.LEFT, padx=(0, 10))
        ttk.Button(button_frame, text="⬇️ Excel'e Aktar", 
                  style="Success.TButton",
                  command=self.export_kar_zarar_excel).pack(side=tk.LEFT)
        
        # Treeview için frame
        tree_frame = ttk.Frame(self.kar_zarar_frame)
        tree_frame.pack(fill=tk.BOTH, expand=True, padx=5, pady=5)
        
        # Scrollbar
        tree_scroll = ttk.Scrollbar(tree_frame)
        tree_scroll.pack(side=tk.RIGHT, fill=tk.Y)
        
        # Treeview
        columns = ("Ürün İsmi", "Alış Mikt.", "Satış Mikt.", "Ort.Alış", "Ort.Satış", 
                  "Kar/Zarar Tutarı", "Kar/Zarar %", "Birim Kar/Zarar")
        self.kar_zarar_tree = ttk.Treeview(tree_frame, columns=columns, show='headings', 
                                          yscrollcommand=tree_scroll.set)
        tree_scroll.config(command=self.kar_zarar_tree.yview)
        
        # Sütun başlıkları ve genişlikleri
        column_widths = {
            "Ürün İsmi": 200, "Alış Mikt.": 80, "Satış Mikt.": 80, 
            "Ort.Alış": 90, "Ort.Satış": 90, "Kar/Zarar Tutarı": 120, 
            "Kar/Zarar %": 100, "Birim Kar/Zarar": 120
        }
        
        for col in columns:
            self.kar_zarar_tree.heading(col, text=col)
            self.kar_zarar_tree.column(col, width=column_widths.get(col, 100), anchor='center')
        
        # Ürün İsmi sütunu sola hizalı
        self.kar_zarar_tree.column("Ürün İsmi", anchor='w')
        
        self.kar_zarar_tree.pack(fill=tk.BOTH, expand=True)
    
    def create_islemler_context_menu(self):
        """İşlemler sağ tık menüsü"""
        self.islemler_context_menu = tk.Menu(self.root, tearoff=0)
        self.islemler_context_menu.add_command(label="✏️ Düzenle", command=self.edit_selected_islem)
        self.islemler_context_menu.add_command(label="🗑️ Sil", command=self.delete_selected_islem)
        self.islemler_context_menu.add_separator()
        self.islemler_context_menu.add_command(label="📋 Kopyala", command=self.copy_selected_islem)
        
        self.islemler_tree.bind('<Button-3>', self.show_islemler_context_menu)
    
    def create_stok_context_menu(self):
        """Stok kartları sağ tık menüsü"""
        self.stok_context_menu = tk.Menu(self.root, tearoff=0)
        self.stok_context_menu.add_command(label="📋 Detay Rapor", command=self.show_selected_urun_detay)
        self.stok_context_menu.add_command(label="✏️ Düzenle", command=self.edit_selected_stok_karti)
        self.stok_context_menu.add_command(label="🗑️ Sil", command=self.delete_selected_stok_karti)
        
        self.stok_tree.bind('<Button-3>', self.show_stok_context_menu)
    
    def create_status_bar(self):
        """Durum çubuğu"""
        self.status_bar = ttk.Frame(self.main_frame)
        self.status_bar.pack(fill=tk.X, side=tk.BOTTOM)
        
        self.status_label = ttk.Label(self.status_bar, text="Hazır", 
                                     foreground="#059669")  # Yeşil renk
        self.status_label.pack(side=tk.LEFT)
        
        # Sağ tarafta tarih/saat - Türkçe formatta
        self.datetime_label = ttk.Label(self.status_bar, text="", 
                                       foreground="#6b7280")  # Gri renk
        self.datetime_label.pack(side=tk.RIGHT)
        self.update_datetime()
    
    def update_datetime(self):
        """Tarih/saat güncelle - Türkçe formatta"""
        now = datetime.now()
        # Türkçe tarih formatı: dd.mm.yyyy hh:mm:ss
        formatted_date = now.strftime("%d.%m.%Y %H:%M:%S")
        self.datetime_label.config(text=formatted_date)
        self.root.after(1000, self.update_datetime)
    
    # EVENT HANDLERs
    def on_tab_changed(self, event):
        """Tab değiştiğinde"""
        self.refresh_all()
    
    def on_islem_double_click(self, event):
        """İşlem satırına çift tık"""
        self.edit_selected_islem()
    
    def on_stok_double_click(self, event):
        """Stok kartı satırına çift tık"""
        self.show_selected_urun_detay()
    
    def show_islemler_context_menu(self, event):
        """İşlemler sağ tık menüsünü göster"""
        if self.islemler_tree.selection():
            self.islemler_context_menu.post(event.x_root, event.y_root)
    
    def show_stok_context_menu(self, event):
        """Stok kartları sağ tık menüsünü göster"""
        if self.stok_tree.selection():
            self.stok_context_menu.post(event.x_root, event.y_root)
    
    def switch_tab(self):
        """Tab değiştir (Ctrl+Tab)"""
        current = self.notebook.select()
        tabs = self.notebook.tabs()
        current_index = tabs.index(current)
        next_index = (current_index + 1) % len(tabs)
        self.notebook.select(tabs[next_index])
    
    # FONT YÖNETİMİ
    def change_font_size(self, delta):
        """Font boyutunu değiştir"""
        new_size = max(10, min(24, self.current_font_size + delta))
        if new_size != self.current_font_size:
            self.current_font_size = new_size
            self.font_size_label.config(text=f"{new_size}px")
            self.stok_manager.save_settings({'fontSize': new_size})
            # Font boyutunu uygulamak için stilleri güncelle
            self.update_fonts()
    
    def update_fonts(self):
        """Font boyutunu GUI'ye uygula"""
        try:
            # Ana stil oluştur
            style = ttk.Style()
            
            # Treeview için font
            style.configure("Treeview", font=('Arial', self.current_font_size))
            style.configure("Treeview.Heading", 
                           font=('Arial', self.current_font_size, 'bold'),
                           background="#e5e7eb",
                           foreground="#000080",  # Lacivert
                           relief="raised",
                           borderwidth=1)
            
            # Label'lar için font
            style.configure("TLabel", font=('Arial', self.current_font_size), foreground="#1a1a1a")  # Koyu gri
            style.configure("TButton", font=('Arial', self.current_font_size))
            style.configure("TEntry", font=('Arial', self.current_font_size))
            style.configure("TCombobox", font=('Arial', self.current_font_size))
            
            # LabelFrame başlıkları - lacivert
            style.configure("TLabelframe.Label", 
                           foreground="#000080",  # Lacivert
                           background="#ffffff",
                           font=('Arial', self.current_font_size, 'bold'))
            
            # Başlık label'ını güncelle - lacivert
            header_font_size = int(self.current_font_size * 1.3)
            style.configure("Header.TLabel", 
                           foreground="#000080",  # Lacivert
                           background="#ffffff",
                           font=('Arial', header_font_size, 'bold'))
            
            # İstatistik label'ları - koyu gri
            stat_font_size = int(self.current_font_size * 1.1)
            style.configure("Stat.TLabel", 
                           foreground="#1a1a1a",  # Çok koyu gri
                           background="#ffffff",
                           font=('Arial', stat_font_size, 'bold'))
            
            # Button stilleri
            style.configure("Success.TButton", 
                           foreground="#000080",  # Lacivert yazı
                           background="#e6f7e6",  # Açık yeşil arka plan
                           font=('Arial', self.current_font_size, 'bold'))
            style.map("Success.TButton",
                     background=[('active', '#d4edda'), ('!active', '#e6f7e6')],
                     foreground=[('active', '#000080'), ('!active', '#000080')])
            
            style.configure("Primary.TButton", 
                           foreground="#000080",  # Lacivert yazı
                           background="#e6f0ff",  # Açık mavi arka plan
                           font=('Arial', self.current_font_size, 'bold'))
            style.map("Primary.TButton",
                     background=[('active', '#cce0ff'), ('!active', '#e6f0ff')],
                     foreground=[('active', '#000080'), ('!active', '#000080')])
            
        except Exception as e:
            print(f"Font güncelleme hatası: {e}")
    
    def refresh_kar_zarar(self):
        """Kar/Zarar verilerini yenile"""
        # Özet verileri al
        ozet = self.stok_manager.get_kar_zarar_ozeti()
        
        # Özet kartları güncelle
        self.kar_zarar_vars['netKarZarar'].set(
            f"{self.stok_manager.format_sayi(ozet['netKarZarar'])} TL"
        )
        self.kar_zarar_vars['toplamKar'].set(
            f"{self.stok_manager.format_sayi(ozet['toplamKar'])} TL"
        )
        self.kar_zarar_vars['toplamZarar'].set(
            f"{self.stok_manager.format_sayi(ozet['toplamZarar'])} TL"
        )
        self.kar_zarar_vars['karliUrunSayisi'].set(str(ozet['karliUrunSayisi']))
        self.kar_zarar_vars['zararliUrunSayisi'].set(str(ozet['zararliUrunSayisi']))
        
        # En karlı/zararlı ürünler
        en_karli = ozet.get('enKarliUrun')
        en_zararli = ozet.get('enZararliUrun')
        
        if en_karli and en_karli['karZararTutari'] > 0:
            self.kar_zarar_vars['enKarliUrun'].set(
                f"{en_karli['urunIsmi']}\n({self.stok_manager.format_sayi(en_karli['karZararTutari'])} TL)"
            )
        else:
            self.kar_zarar_vars['enKarliUrun'].set("-")
        
        if en_zararli and en_zararli['karZararTutari'] < 0:
            self.kar_zarar_vars['enZararliUrun'].set(
                f"{en_zararli['urunIsmi']}\n({self.stok_manager.format_sayi(en_zararli['karZararTutari'])} TL)"
            )
        else:
            self.kar_zarar_vars['enZararliUrun'].set("-")
        
        # Tablo verilerini güncelle
        # Mevcut verileri temizle
        for item in self.kar_zarar_tree.get_children():
            self.kar_zarar_tree.delete(item)
        
        # Yeni verileri ekle
        kar_zarar_data = self.stok_manager.calculate_kar_zarar_analizi()
        
        if not kar_zarar_data:
            # Boş mesajı
            self.kar_zarar_tree.insert("", tk.END, values=(
                "Henüz kar/zarar analizi yok", "", "", "", "", "", "", "İşlem yapın ve analizi yenileyin"
            ))
        else:
            for item in kar_zarar_data:
                # Kar/zarar tutarı için renk gösterimi
                kar_zarar_tutar = item['karZararTutari']
                kar_zarar_text = self.stok_manager.format_sayi(kar_zarar_tutar)
                if kar_zarar_tutar > 0:
                    kar_zarar_text = f"✅ {kar_zarar_text}"
                elif kar_zarar_tutar < 0:
                    kar_zarar_text = f"❌ {kar_zarar_text}"
                else:
                    kar_zarar_text = f"⚪ {kar_zarar_text}"
                
                # Yüzde formatı
                yuzde_text = f"{item['karZararYuzdesi']:+.1f}%"
                if item['karZararYuzdesi'] > 0:
                    yuzde_text = f"✅ {yuzde_text}"
                elif item['karZararYuzdesi'] < 0:
                    yuzde_text = f"❌ {yuzde_text}"
                else:
                    yuzde_text = f"⚪ {yuzde_text}"
                
                # Birim kar/zarar
                birim_kar_zarar = item['birimKarZarar']
                birim_text = f"{self.stok_manager.format_sayi(birim_kar_zarar)}"
                if birim_kar_zarar > 0:
                    birim_text = f"✅ +{birim_text}"
                elif birim_kar_zarar < 0:
                    birim_text = f"❌ {birim_text}"
                else:
                    birim_text = f"⚪ {birim_text}"
                
                values = (
                    item['urunIsmi'],
                    f"{self.stok_manager.format_sayi_basit(item['toplamAlisMiktari'])} {item['birim']}",
                    f"{self.stok_manager.format_sayi_basit(item['toplamSatisMiktari'])} {item['birim']}",
                    self.stok_manager.format_sayi(item['ortAlisFiyati']),
                    self.stok_manager.format_sayi(item['ortSatisFiyati']),
                    kar_zarar_text,
                    yuzde_text,
                    birim_text
                )
                
                self.kar_zarar_tree.insert("", tk.END, values=values)
    
    def export_kar_zarar_excel(self):
        """Kar/Zarar analizini Excel'e aktar"""
        try:
            filename = filedialog.asksaveasfilename(
                defaultextension=".csv",
                filetypes=[("CSV files", "*.csv")],
                title="Kar/Zarar Analizi Excel'e Aktar"
            )
            
            if filename:
                exporter = ReportExporter(self.stok_manager)
                exporter.export_kar_zarar_csv(filename)
                messagebox.showinfo("Başarılı", "Kar/Zarar analizi Excel dosyası oluşturuldu!")
        except Exception as e:
            messagebox.showerror("Hata", f"Excel dışa aktarma hatası: {e}")
    
    # VERİ YENİLEME
    def refresh_all(self):
        """Tüm veriyi yenile"""
        self.refresh_stats()
        self.refresh_islemler()
        self.refresh_stok_kartlari()
        self.refresh_kar_zarar()
        self.status_label.config(text="Veriler yenilendi", foreground="#059669")
        self.root.after(3000, lambda: self.status_label.config(text="Hazır", foreground="#059669"))
    
    def refresh_stats(self):
        """İstatistikleri yenile"""
        stats = self.stok_manager.get_statistics()
        self.stats_vars['totalProducts'].set(str(stats['totalProducts']))
        self.stats_vars['totalValue'].set(self.stok_manager.format_sayi(stats['totalValue']) + " TL")
        self.stats_vars['totalEntries'].set(str(stats['totalEntries']))
        self.stats_vars['totalExits'].set(str(stats['totalExits']))
    
    def refresh_islemler(self):
        """İşlemler tablosunu yenile"""
        # Mevcut verileri temizle
        for item in self.islemler_tree.get_children():
            self.islemler_tree.delete(item)
        
        # Yeni verileri ekle
        islemler = self.stok_manager.get_all_islemler(sort_by_date=True)
        
        if not islemler:
            # Boş mesajı
            self.islemler_tree.insert("", tk.END, values=(
                "Henüz işlem yok", "", "", "Yeni İşlem Ekle butonunu kullanın", "", "", ""
            ))
        else:
            for islem in islemler:
                # Tür için badge benzeri gösterim
                tur_text = f"🟢 {islem['islemTuru']}" if islem['islemTuru'] == 'GİRİŞ' else f"🔴 {islem['islemTuru']}"
                
                # Tarih formatını Türkçe'ye çevir (aynı yıl kontrolü ile)
                tarih_formatted = self.stok_manager.format_tarih(islem['tarih'], goster_yil=False)
                
                values = (
                    tur_text,
                    tarih_formatted,
                    islem['faturaNo'],
                    islem['urunIsim'],
                    f"{self.stok_manager.format_sayi_basit(islem['miktar'])} {islem.get('birim', 'ADET')}",
                    self.stok_manager.format_sayi(islem['fiyat']),
                    self.stok_manager.format_sayi(islem['tutar'])
                )
                
                item_id = self.islemler_tree.insert("", tk.END, values=values, tags=(str(islem['id']),))
    
    def refresh_stok_kartlari(self):
        """Stok kartları tablosunu yenile"""
        # Mevcut verileri temizle
        for item in self.stok_tree.get_children():
            self.stok_tree.delete(item)
        
        # Yeni verileri ekle
        stok_kartlari = self.stok_manager.hesapla_stok_kartlari()
        
        if not stok_kartlari:
            # Boş mesajı
            self.stok_tree.insert("", tk.END, values=(
                "", "Henüz stok kartı yok", "", "", "", "", "", "Yeni Stok Kartı butonunu kullanın"
            ))
        else:
            for stok in stok_kartlari:
                # Kalan stok için renk gösterimi
                kalan_stok = stok.get('kalanStok', 0)
                kalan_text = self.stok_manager.format_sayi_basit(kalan_stok)
                if kalan_stok > 0:
                    kalan_text = f"✅ {kalan_text}"
                elif kalan_stok < 0:
                    kalan_text = f"❌ {kalan_text}"
                else:
                    kalan_text = f"⚪ {kalan_text}"
                
                values = (
                    stok['kdo'],
                    stok['urunIsmi'],
                    stok['birim'],
                    self.stok_manager.format_sayi_basit(stok.get('girisMiktari', 0)),
                    self.stok_manager.format_sayi_basit(stok.get('cikisMiktari', 0)),
                    kalan_text,
                    self.stok_manager.format_sayi(stok.get('girisTutari', 0)),
                    self.stok_manager.format_sayi(stok.get('cikisTutari', 0))
                )
                
                item_id = self.stok_tree.insert("", tk.END, values=values, tags=(stok['kdo'],))
    
    # DIALOG AÇMA FONKSİYONLARI
    def open_islem_dialog(self, islem_id=None):
        """İşlem dialog'unu aç"""
        dialog = IslemDialog(self.root, self.stok_manager, islem_id)
        if dialog.result:
            self.refresh_all()
    
    def open_stok_karti_dialog(self, kdo=None):
        """Stok kartı dialog'unu aç"""
        dialog = StokKartiDialog(self.root, self.stok_manager, kdo)
        if dialog.result:
            self.refresh_all()
    
    def open_fifo_rapor_dialog(self):
        """FIFO rapor dialog'unu aç"""
        FifoRaporDialog(self.root, self.stok_manager)
    
    # SEÇİLİ ÖĞELERLE İŞLEMLER
    def edit_selected_islem(self):
        """Seçili işlemi düzenle"""
        selection = self.islemler_tree.selection()
        if not selection:
            messagebox.showwarning("Uyarı", "Lütfen düzenlenecek işlemi seçin!")
            return
        
        item = selection[0]
        # Tag'den ID'yi al
        try:
            tags = self.islemler_tree.item(item, 'tags')
            if tags:
                islem_id = int(tags[0])
                self.open_islem_dialog(islem_id)
            else:
                raise ValueError("Tag bulunamadı")
        except (ValueError, TypeError, IndexError):
            # Tag yoksa, satır indeksinden ID'yi bul
            try:
                row_index = self.islemler_tree.index(item)
                islemler = self.stok_manager.get_all_islemler(sort_by_date=True)
                if row_index < len(islemler):
                    islem_id = islemler[row_index]['id']
                    self.open_islem_dialog(islem_id)
            except:
                messagebox.showerror("Hata", "İşlem ID'si bulunamadı!")
    
    def delete_selected_islem(self):
        """Seçili işlemi sil"""
        selection = self.islemler_tree.selection()
        if not selection:
            messagebox.showwarning("Uyarı", "Lütfen silinecek işlemi seçin!")
            return
        
        if messagebox.askyesno("Onay", "Seçili işlemi silmek istediğinizden emin misiniz?"):
            try:
                item = selection[0]
                # Tag'den ID'yi al
                try:
                    tags = self.islemler_tree.item(item, 'tags')
                    if tags:
                        islem_id = int(tags[0])
                    else:
                        raise ValueError("Tag bulunamadı")
                except (ValueError, TypeError, IndexError):
                    # Tag yoksa, satır indeksinden ID'yi bul
                    row_index = self.islemler_tree.index(item)
                    islemler = self.stok_manager.get_all_islemler(sort_by_date=True)
                    if row_index < len(islemler):
                        islem_id = islemler[row_index]['id']
                    else:
                        raise ValueError("İşlem bulunamadı")
                
                self.stok_manager.delete_islem(islem_id)
                self.stok_manager.save_data()
                self.refresh_all()
                messagebox.showinfo("Başarılı", "İşlem silindi!")
                
            except Exception as e:
                messagebox.showerror("Hata", f"İşlem silinemedi: {e}")
    
    def copy_selected_islem(self):
        """Seçili işlemi kopyala (gelecek geliştirme için)"""
        messagebox.showinfo("Bilgi", "Kopyalama özelliği yakında eklenecek!")
    
    def edit_selected_stok_karti(self):
        """Seçili stok kartını düzenle"""
        selection = self.stok_tree.selection()
        if not selection:
            messagebox.showwarning("Uyarı", "Lütfen düzenlenecek stok kartını seçin!")
            return
        
        item = selection[0]
        # Tag'den KDO'yu al
        try:
            tags = self.stok_tree.item(item, 'tags')
            if tags:
                kdo = tags[0]
                self.open_stok_karti_dialog(kdo)
            else:
                raise ValueError("Tag bulunamadı")
        except (ValueError, TypeError, IndexError):
            # Tag yoksa, values'den KDO'yu al
            kdo = self.stok_tree.item(item)['values'][0]
            self.open_stok_karti_dialog(kdo)
    
    def delete_selected_stok_karti(self):
        """Seçili stok kartını sil"""
        selection = self.stok_tree.selection()
        if not selection:
            messagebox.showwarning("Uyarı", "Lütfen silinecek stok kartını seçin!")
            return
        
        item = selection[0]
        # Tag'den KDO'yu al
        try:
            tags = self.stok_tree.item(item, 'tags')
            if tags:
                kdo = tags[0]
            else:
                raise ValueError("Tag bulunamadı")
        except (ValueError, TypeError, IndexError):
            # Tag yoksa, values'den KDO'yu al
            kdo = self.stok_tree.item(item)['values'][0]
        
        if messagebox.askyesno("Onay", f"'{kdo}' kodlu stok kartını silmek istediğinizden emin misiniz?"):
            try:
                self.stok_manager.delete_stok_karti(kdo)
                self.stok_manager.save_data()
                self.refresh_all()
                messagebox.showinfo("Başarılı", "Stok kartı silindi!")
            except Exception as e:
                messagebox.showerror("Hata", f"Stok kartı silinemedi: {e}")
    
    def show_selected_urun_detay(self):
        """Seçili ürünün detay raporunu göster"""
        selection = self.stok_tree.selection()
        if not selection:
            messagebox.showwarning("Uyarı", "Lütfen detayını görmek istediğiniz ürünü seçin!")
            return
        
        item = selection[0]
        # Ürün ismini values'den al (bu görünür olmalı)
        urun_ismi = self.stok_tree.item(item)['values'][1]
        UrunDetayDialog(self.root, self.stok_manager, urun_ismi)
    
    # SIRALAMA FONKSİYONLARI
    def sort_islemler(self, column):
        """İşlemler tablosunu sırala"""
        # Gelecek geliştirme için placeholder
        pass
    
    def sort_stok(self, column):
        """Stok tablosunu sırala"""
        # Gelecek geliştirme için placeholder
        pass
    
    # İÇE/DIŞA AKTARMA FONKSİYONLARI
    def export_data(self):
        """Tüm veriyi JSON olarak dışa aktar"""
        try:
            filename = filedialog.asksaveasfilename(
                defaultextension=".json",
                filetypes=[("JSON files", "*.json")],
                title="Veriyi Dışa Aktar"
            )
            
            if filename:
                data = self.stok_manager.export_all_data()
                with open(filename, 'w', encoding='utf-8') as f:
                    json.dump(data, f, ensure_ascii=False, indent=2)
                messagebox.showinfo("Başarılı", "Veriler başarıyla dışa aktarıldı!")
        except Exception as e:
            messagebox.showerror("Hata", f"Dışa aktarma hatası: {e}")
    
    def import_data(self):
        """JSON dosyasından veri içe aktar"""
        try:
            filename = filedialog.askopenfilename(
                filetypes=[("JSON files", "*.json")],
                title="Veri İçe Aktar"
            )
            
            if filename:
                with open(filename, 'r', encoding='utf-8') as f:
                    data = json.load(f)
                
                if self.stok_manager.import_all_data(data):
                    self.stok_manager.save_data()
                    self.refresh_all()
                    messagebox.showinfo("Başarılı", "Veriler başarıyla içe aktarıldı!")
                else:
                    messagebox.showerror("Hata", "Geçersiz dosya formatı!")
        except Exception as e:
            messagebox.showerror("Hata", f"İçe aktarma hatası: {e}")
    
    def export_stok_kartlari_excel(self):
        """Stok kartlarını Excel'e aktar"""
        try:
            filename = filedialog.asksaveasfilename(
                defaultextension=".csv",
                filetypes=[("CSV files", "*.csv")],
                title="Stok Kartları Excel'e Aktar"
            )
            
            if filename:
                exporter = ReportExporter(self.stok_manager)
                exporter.export_stok_kartlari_csv(filename)
                messagebox.showinfo("Başarılı", "Stok kartları Excel dosyası oluşturuldu!")
        except Exception as e:
            messagebox.showerror("Hata", f"Excel dışa aktarma hatası: {e}")
    
    def export_islemler_excel(self):
        """İşlemleri Excel'e aktar"""
        try:
            filename = filedialog.asksaveasfilename(
                defaultextension=".csv",
                filetypes=[("CSV files", "*.csv")],
                title="İşlemler Excel'e Aktar"
            )
            
            if filename:
                exporter = ReportExporter(self.stok_manager)
                exporter.export_islemler_csv(filename)
                messagebox.showinfo("Başarılı", "İşlemler Excel dosyası oluşturuldu!")
        except Exception as e:
            messagebox.showerror("Hata", f"Excel dışa aktarma hatası: {e}")