#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
Dialog Pencereleri - Stok Takip Uygulaması
Gelişmiş klavye navigasyonu ve Türkçe tarih formatı
Virgül (,) ondalık desteği eklendi
"""

import tkinter as tk
from tkinter import ttk, messagebox, filedialog
from datetime import datetime, date
import re

class TurkishDateEntry(ttk.Frame):
    """Türkçe tarih giriş widget'ı - DD.MM.YYYY formatında - Gelişmiş navigasyon"""
    def __init__(self, parent, initial_date=None, **kwargs):
        super().__init__(parent, **kwargs)
        
        # Parent dialog referansı
        self.parent_dialog = None
        
        # Varsayılan tarih
        if initial_date:
            try:
                if isinstance(initial_date, str):
                    if '-' in initial_date:  # YYYY-MM-DD format
                        dt = datetime.strptime(initial_date, '%Y-%m-%d')
                    else:  # DD.MM.YYYY format
                        dt = datetime.strptime(initial_date, '%d.%m.%Y')
                else:
                    dt = initial_date
            except:
                dt = datetime.now()
        else:
            dt = datetime.now()
        
        # Entry'ler için değişkenler
        self.day_var = tk.StringVar(value=f"{dt.day:02d}")
        self.month_var = tk.StringVar(value=f"{dt.month:02d}")
        self.year_var = tk.StringVar(value=str(dt.year))
        
        # Layout
        self.create_widgets()
        self.setup_navigation()
    
    def set_parent_dialog(self, parent_dialog):
        """Parent dialog referansını ayarla"""
        self.parent_dialog = parent_dialog
    
    def create_widgets(self):
        """Widget'ları oluştur"""
        # Gün
        self.day_entry = ttk.Entry(self, textvariable=self.day_var, width=3, justify='center')
        self.day_entry.pack(side=tk.LEFT)
        
        # Nokta
        ttk.Label(self, text=".").pack(side=tk.LEFT)
        
        # Ay
        self.month_entry = ttk.Entry(self, textvariable=self.month_var, width=3, justify='center')
        self.month_entry.pack(side=tk.LEFT)
        
        # Nokta
        ttk.Label(self, text=".").pack(side=tk.LEFT)
        
        # Yıl
        self.year_entry = ttk.Entry(self, textvariable=self.year_var, width=5, justify='center')
        self.year_entry.pack(side=tk.LEFT)
    
    def setup_navigation(self):
        """Klavye navigasyonunu ayarla"""
        try:
            # Gün alanı
            self.day_var.trace('w', self.on_day_change)
            self.day_entry.bind('<KeyPress>', self.on_day_keypress)
            self.day_entry.bind('<FocusIn>', lambda e: self.safe_select_range(self.day_entry))
            
            # Ay alanı  
            self.month_var.trace('w', self.on_month_change)
            self.month_entry.bind('<KeyPress>', self.on_month_keypress)
            self.month_entry.bind('<FocusIn>', lambda e: self.safe_select_range(self.month_entry))
            
            # Yıl alanı
            self.year_entry.bind('<FocusIn>', lambda e: self.safe_select_range(self.year_entry))
            self.year_entry.bind('<KeyPress>', self.on_year_keypress)
            
            # Tüm alanlar için ortak navigasyon
            for entry in [self.day_entry, self.month_entry, self.year_entry]:
                try:
                    entry.bind('<Tab>', self.on_tab)
                    entry.bind('<Return>', self.on_enter)
                    entry.bind('<Down>', self.on_down)
                    entry.bind('<Up>', self.on_up)
                    entry.bind('<BackSpace>', self.on_backspace)
                except Exception as e:
                    print(f"Entry binding hatası: {e}")
        except Exception as e:
            print(f"Navigation setup hatası: {e}")
    
    def safe_select_range(self, entry):
        """Güvenli metin seçimi"""
        try:
            if entry and entry.winfo_exists():
                entry.select_range(0, tk.END)
        except Exception as e:
            print(f"Select range hatası: {e}")
    
    def on_day_change(self, *args):
        """Gün değiştiğinde"""
        try:
            day_text = self.day_var.get()
            if len(day_text) >= 2:
                day_num = int(day_text)
                if 1 <= day_num <= 31:
                    # Otomatik ay alanına geç
                    self.month_entry.focus_set()
                    self.month_entry.select_range(0, tk.END)
        except ValueError:
            pass
        except Exception as e:
            print(f"Day change hatası: {e}")
    
    def on_month_change(self, *args):
        """Ay değiştiğinde"""
        try:
            month_text = self.month_var.get()
            if len(month_text) >= 2:
                month_num = int(month_text)
                if 1 <= month_num <= 12:
                    # Yıl alanına geç
                    self.year_entry.focus_set()
                    self.year_entry.select_range(0, tk.END)
        except ValueError:
            pass
        except Exception as e:
            print(f"Month change hatası: {e}")
    
    def on_day_keypress(self, event):
        """Gün alanı tuş basımı"""
        if event.char.isdigit():
            return None
        elif event.keysym in ['BackSpace', 'Delete', 'Tab', 'Return', 'Down', 'Up', 'Left', 'Right']:
            return None
        else:
            return 'break'
    
    def on_month_keypress(self, event):
        """Ay alanı tuş basımı"""
        if event.char.isdigit():
            return None
        elif event.keysym in ['BackSpace', 'Delete', 'Tab', 'Return', 'Down', 'Up', 'Left', 'Right']:
            return None
        else:
            return 'break'
    
    def on_year_keypress(self, event):
        """Yıl alanı tuş basımı"""
        if event.char.isdigit():
            return None
        elif event.keysym in ['BackSpace', 'Delete', 'Tab', 'Return', 'Down', 'Up', 'Left', 'Right']:
            return None
        else:
            return 'break'
    
    def on_tab(self, event):
        """Tab tuşu"""
        if event.widget == self.day_entry:
            self.month_entry.focus_set()
            self.month_entry.select_range(0, tk.END)
            return 'break'
        elif event.widget == self.month_entry:
            self.year_entry.focus_set()
            self.year_entry.select_range(0, tk.END)
            return 'break'
        elif event.widget == self.year_entry:
            self.advance_to_next_field()
            return 'break'
        return None
    
    def on_enter(self, event):
        """Enter tuşu"""
        if event.widget == self.day_entry:
            self.month_entry.focus_set()
            self.month_entry.select_range(0, tk.END)
            return 'break'
        elif event.widget == self.month_entry:
            self.year_entry.focus_set()
            self.year_entry.select_range(0, tk.END)
            return 'break'
        elif event.widget == self.year_entry:
            self.advance_to_next_field()
            return 'break'
        return None
    
    def on_down(self, event):
        """Aşağı ok tuşu"""
        return self.on_enter(event)
    
    def on_up(self, event):
        """Yukarı ok tuşu"""
        if event.widget == self.year_entry:
            self.month_entry.focus_set()
            self.month_entry.icursor(tk.END)
            return 'break'
        elif event.widget == self.month_entry:
            self.day_entry.focus_set()
            self.day_entry.icursor(tk.END)
            return 'break'
        return None
    
    def on_backspace(self, event):
        """Backspace tuşu"""
        current_text = event.widget.get()
        if not current_text:
            if event.widget == self.month_entry:
                self.day_entry.focus_set()
                self.day_entry.icursor(tk.END)
                return 'break'
            elif event.widget == self.year_entry:
                self.month_entry.focus_set()
                self.month_entry.icursor(tk.END)
                return 'break'
        return None
    
    def advance_to_next_field(self):
        """Bir sonraki alana geç"""
        try:
            if self.parent_dialog and hasattr(self.parent_dialog, 'navigation_widgets'):
                for i, widget in enumerate(self.parent_dialog.navigation_widgets):
                    if widget == self:
                        if i + 1 < len(self.parent_dialog.navigation_widgets):
                            next_widget = self.parent_dialog.navigation_widgets[i + 1]
                            next_widget.focus_set()
                            if hasattr(next_widget, 'select_range'):
                                try:
                                    next_widget.select_range(0, tk.END)
                                except:
                                    pass
                            return
                if hasattr(self.parent_dialog, 'fatura_entry'):
                    self.parent_dialog.fatura_entry.focus_set()
                    if hasattr(self.parent_dialog.fatura_entry, 'select_range'):
                        try:
                            self.parent_dialog.fatura_entry.select_range(0, tk.END)
                        except:
                            pass
        except Exception as e:
            print(f"Advance to next field hatası: {e}")
    
    def get_date(self):
        """Tarihi DD.MM.YYYY formatında döndür"""
        try:
            day_str = self.day_var.get().strip()
            month_str = self.month_var.get().strip()
            year_str = self.year_var.get().strip()
            
            if not day_str or not month_str or not year_str:
                raise ValueError("Boş alan var")
            
            day = int(day_str)
            month = int(month_str)
            year = int(year_str)
            
            test_date = datetime(year, month, day)
            
            return f"{day:02d}.{month:02d}.{year}"
        except ValueError as e:
            raise ValueError(f"Geçersiz tarih formatı: {e}")
        except Exception as e:
            print(f"get_date hatası: {e}")
            raise ValueError(f"Geçersiz tarih formatı: {e}")
    
    def get_iso_date(self):
        """Tarihi YYYY-MM-DD formatında döndür"""
        try:
            day_str = self.day_var.get().strip()
            month_str = self.month_var.get().strip()
            year_str = self.year_var.get().strip()
            
            if not day_str or not month_str or not year_str:
                raise ValueError("Boş alan var")
            
            day = int(day_str)
            month = int(month_str)
            year = int(year_str)
            
            test_date = datetime(year, month, day)
            
            return f"{year}-{month:02d}-{day:02d}"
        except ValueError as e:
            raise ValueError(f"Geçersiz tarih formatı: {e}")
        except Exception as e:
            print(f"get_iso_date hatası: {e}")
            raise ValueError(f"Geçersiz tarih formatı: {e}")
    
    def set_date(self, date_str):
        """Tarihi ayarla"""
        try:
            if not date_str:
                today = datetime.now()
                self.day_var.set(f"{today.day:02d}")
                self.month_var.set(f"{today.month:02d}")
                self.year_var.set(str(today.year))
                return
            
            date_str = str(date_str).strip()
            dt = None
            
            if '-' in date_str and len(date_str) == 10:
                try:
                    dt = datetime.strptime(date_str, '%Y-%m-%d')
                except:
                    pass
            
            elif '.' in date_str and len(date_str) == 10:
                try:
                    dt = datetime.strptime(date_str, '%d.%m.%Y')
                except:
                    pass
            
            elif 'T' in date_str:
                try:
                    dt = datetime.fromisoformat(date_str.replace('Z', '+00:00'))
                except:
                    pass
            
            if dt is None:
                dt = datetime.now()
                print(f"Tarih parse edilemedi, bugünün tarihi kullanılıyor: {date_str}")
            
            self.day_var.set(f"{dt.day:02d}")
            self.month_var.set(f"{dt.month:02d}")
            self.year_var.set(str(dt.year))
            
        except Exception as e:
            print(f"set_date hatası: {e}, input: {date_str}")
            today = datetime.now()
            self.day_var.set(f"{today.day:02d}")
            self.month_var.set(f"{today.month:02d}")
            self.year_var.set(str(today.year))

class BaseDialog:
    """Temel dialog sınıfı"""
    def __init__(self, parent, title, geometry="600x400"):
        self.parent = parent
        self.result = None
        
        self.dialog = tk.Toplevel(parent)
        self.dialog.title(title)
        self.dialog.geometry(geometry)
        self.dialog.resizable(False, False)
        self.dialog.transient(parent)
        self.dialog.grab_set()
        
        self.center_window()
        
        self.dialog.bind('<Escape>', lambda e: self.cancel())
        
        self.dialog.focus_set()
    
    def center_window(self):
        """Pencereyi ortala"""
        self.dialog.update_idletasks()
        x = (self.dialog.winfo_screenwidth() // 2) - (self.dialog.winfo_width() // 2)
        y = (self.dialog.winfo_screenheight() // 2) - (self.dialog.winfo_height() // 2)
        self.dialog.geometry(f"+{x}+{y}")
    
    def cancel(self):
        """İptal et"""
        self.result = None
        self.dialog.destroy()
    
    def ok(self):
        """Tamam"""
        self.result = True
        self.dialog.destroy()

class IslemDialog(BaseDialog):
    """İşlem ekleme/düzenleme dialog'u - Türkçe ondalık desteği ile"""
    def __init__(self, parent, stok_manager, islem_id=None):
        self.stok_manager = stok_manager
        self.islem_id = islem_id
        self.continuous_entry = False
        self.last_form_data = None
        self.dialog_closing = False
        
        self.navigation_widgets = []
        
        title = "İşlemi Düzenle" if islem_id else "Yeni İşlem Ekle"
        super().__init__(parent, title, "650x520")
        
        self.create_widgets(self.dialog)
        self.setup_keyboard_navigation()
        self.load_data()
        
        if not self.islem_id:
            self.dialog.after(100, lambda: self.islem_turu_combo.focus_set())
        
        self.dialog.wait_window()
    
    def create_widgets(self, parent):
        """Widget'ları oluştur"""
        main_frame = ttk.Frame(self.dialog)
        main_frame.pack(fill=tk.BOTH, expand=True, padx=20, pady=20)
        
        self.create_form_fields(main_frame)
        self.create_stock_display(main_frame)
        self.create_buttons(main_frame)
    
    def create_form_fields(self, parent):
        """Form alanlarını oluştur"""
        # İşlem türü
        ttk.Label(parent, text="İşlem Türü:").grid(row=0, column=0, sticky="w", pady=5)
        self.islem_turu_var = tk.StringVar(value="GİRİŞ")
        self.islem_turu_combo = ttk.Combobox(parent, textvariable=self.islem_turu_var, 
                                       values=["GİRİŞ", "ÇIKIŞ"], state="readonly", width=40)
        self.islem_turu_combo.grid(row=0, column=1, sticky="ew", pady=5)
        self.islem_turu_combo.bind('<<ComboboxSelected>>', self.on_islem_turu_changed)
        self.navigation_widgets.append(self.islem_turu_combo)
        
        # Tarih
        ttk.Label(parent, text="Tarih (Gün.Ay.Yıl):").grid(row=1, column=0, sticky="w", pady=5)
        self.tarih_entry = TurkishDateEntry(parent)
        self.tarih_entry.set_parent_dialog(self)
        self.tarih_entry.grid(row=1, column=1, sticky="w", pady=5)
        self.navigation_widgets.append(self.tarih_entry) 
        
        for var in [self.tarih_entry.day_var, self.tarih_entry.month_var, self.tarih_entry.year_var]:
            var.trace('w', self.on_tarih_changed)
        
        # Fatura No
        ttk.Label(parent, text="Fatura No / Belge No:").grid(row=2, column=0, sticky="w", pady=5)
        self.fatura_no_var = tk.StringVar()
        self.fatura_entry = ttk.Entry(parent, textvariable=self.fatura_no_var, width=42)
        self.fatura_entry.grid(row=2, column=1, sticky="ew", pady=5)
        self.navigation_widgets.append(self.fatura_entry)
        
        # Ürün İsmi
        ttk.Label(parent, text="Ürün İsmi / Kodu:").grid(row=3, column=0, sticky="w", pady=5)
        self.urun_isim_var = tk.StringVar()
        self.urun_entry = ttk.Entry(parent, textvariable=self.urun_isim_var, width=42)
        self.urun_entry.grid(row=3, column=1, sticky="ew", pady=5)
        self.urun_entry.bind('<KeyRelease>', self.on_urun_search)
        self.urun_entry.bind('<FocusIn>', self.on_urun_focus)
        self.urun_entry.bind('<Down>', self.on_urun_down)
        self.navigation_widgets.append(self.urun_entry)
        
        # Öneriler listbox
        self.suggestions_frame = ttk.Frame(parent)
        self.suggestions_frame.grid(row=4, column=1, sticky="ew", pady=5)
        
        self.suggestions_listbox = tk.Listbox(self.suggestions_frame, height=4, exportselection=False)
        self.suggestions_listbox.pack(fill=tk.BOTH, expand=True)
        self.suggestions_listbox.bind('<<ListboxSelect>>', self._on_listbox_selection_changed)
        self.suggestions_listbox.bind('<Return>', self.on_suggestion_enter)
        self.suggestions_listbox.bind('<Button-1>', self.on_suggestion_click)
        self.suggestions_listbox.bind('<Tab>', self.on_suggestion_tab)
        self.suggestions_listbox.bind('<Down>', self.on_suggestion_down)
        self.suggestions_listbox.bind('<Up>', self.on_suggestion_up)
        self.suggestions_listbox.bind('<Escape>', self.hide_suggestions)
        self.suggestions_listbox.pack_forget()
        
        # Miktar
        ttk.Label(parent, text="Miktar:").grid(row=5, column=0, sticky="w", pady=5)
        self.miktar_var = tk.StringVar()
        self.miktar_entry = ttk.Entry(parent, textvariable=self.miktar_var, width=42)
        self.miktar_entry.grid(row=5, column=1, sticky="ew", pady=5)
        self.miktar_entry.bind('<KeyRelease>', self.validate_number)
        self.navigation_widgets.append(self.miktar_entry)
        
        # Fiyat
        ttk.Label(parent, text="Birim Fiyat:").grid(row=6, column=0, sticky="w", pady=5)
        self.fiyat_var = tk.StringVar()
        self.fiyat_entry = ttk.Entry(parent, textvariable=self.fiyat_var, width=42)
        self.fiyat_entry.grid(row=6, column=1, sticky="ew", pady=5)
        self.fiyat_entry.bind('<KeyRelease>', self.validate_number)
        self.navigation_widgets.append(self.fiyat_entry)
        
        parent.columnconfigure(1, weight=1)
    
    def validate_number(self, event=None):
        """Sayı validasyonu - Rakam, nokta ve virgül kabul et (Türkçe format)"""
        current_widget = self.dialog.focus_get()
        if current_widget in [self.miktar_entry, self.fiyat_entry]:
            var = self.miktar_var if current_widget == self.miktar_entry else self.fiyat_var
            
            new_value = var.get()
            if not new_value:
                return
            
            # Sadece rakam, nokta ve virgül kabul et
            if re.fullmatch(r"^\d*[.,]?\d*$", new_value):
                # Tek ondalık ayırıcı kontrolü
                decimal_count = new_value.count('.') + new_value.count(',')
                if decimal_count > 1:
                    var.set(new_value[:-1])
            else:
                # Geçersiz karakter, son karakteri sil
                var.set(new_value[:-1])
    
    def create_buttons(self, parent):
        """Butonları oluştur"""
        button_frame = ttk.Frame(parent)
        button_frame.grid(row=8, column=0, columnspan=2, pady=20)
        
        if not self.islem_id:
            self.save_continue_btn = ttk.Button(button_frame, text="Kaydet ve Devam Et", 
                                               command=self.save_and_continue)
            self.save_continue_btn.pack(side=tk.LEFT, padx=5)
            self.navigation_widgets.append(self.save_continue_btn)
            
            self.new_invoice_btn = ttk.Button(button_frame, text="Yeni Fatura", 
                                             command=self.start_new_invoice)
            self.new_invoice_btn.pack(side=tk.LEFT, padx=5)
            self.new_invoice_btn.pack_forget()
            self.navigation_widgets.append(self.new_invoice_btn)
        
        save_text = "Güncelle" if self.islem_id else "Kaydet"
        self.save_btn = ttk.Button(button_frame, text=save_text, command=self.save)
        self.save_btn.pack(side=tk.LEFT, padx=5)
        self.navigation_widgets.append(self.save_btn)
        
        self.cancel_btn = ttk.Button(button_frame, text="İptal", command=self.cancel)
        self.cancel_btn.pack(side=tk.LEFT, padx=5)
        self.navigation_widgets.append(self.cancel_btn)
        
        self.dialog.bind('<Control-Return>', lambda e: self.save())
        self.dialog.bind('<F10>', lambda e: self.save_and_continue() if not self.islem_id else None)
    
    def setup_keyboard_navigation(self):
        """Klavye navigasyonunu ayarla"""
        date_entry_widgets = [self.tarih_entry.day_entry, self.tarih_entry.month_entry, self.tarih_entry.year_entry]
        for i, widget in enumerate(self.navigation_widgets):
            if widget not in date_entry_widgets and widget != self.suggestions_listbox:
                widget.bind('<Tab>', lambda e, idx=i: self.safe_navigate_next(idx))
                widget.bind('<Return>', lambda e, idx=i: self.safe_navigate_next(idx))
                widget.bind('<Down>', lambda e, idx=i: self.safe_navigate_next(idx))
                widget.bind('<Shift-Tab>', lambda e, idx=i: self.safe_navigate_prev(idx))
                widget.bind('<Up>', lambda e, idx=i: self.safe_navigate_prev(idx))
        
        self.urun_entry.bind('<Return>', self.on_urun_enter)
        self.urun_entry.bind('<Down>', self.on_urun_down)
    
    def safe_navigate_next(self, current_index):
        """Güvenli sonraki widget navigasyonu"""
        try:
            return self.navigate_to_next_widget(current_index)
        except Exception as e:
            print(f"Navigasyon hatası (next): {e}")
            return 'break'
    
    def safe_navigate_prev(self, current_index):
        """Güvenli önceki widget navigasyonu"""
        try:
            return self.navigate_to_prev_widget(current_index)
        except Exception as e:
            print(f"Navigasyon hatası (prev): {e}")
            return 'break'
    
    def navigate_to_next_widget(self, current_index):
        """Bir sonraki widget'a geç"""
        if self.suggestions_listbox.winfo_viewable() and self.dialog.focus_get() not in [self.urun_entry, self.suggestions_listbox]:
            self.hide_suggestions()

        try:
            next_widget_index = current_index + 1
            
            if self.navigation_widgets[current_index] == self.islem_turu_combo:
                self.tarih_entry.day_entry.focus_set()
                self.tarih_entry.day_entry.select_range(0, tk.END)
                return 'break'
            
            if next_widget_index < len(self.navigation_widgets) and self.navigation_widgets[next_widget_index] == self.tarih_entry:
                self.tarih_entry.day_entry.focus_set()
                self.tarih_entry.day_entry.select_range(0, tk.END)
                return 'break'

            if next_widget_index < len(self.navigation_widgets):
                next_widget = self.navigation_widgets[next_widget_index]
                
                if not next_widget.winfo_viewable():
                    if hasattr(self, 'save_continue_btn') and next_widget == self.save_continue_btn and not self.save_continue_btn.winfo_viewable():
                        if hasattr(self, 'new_invoice_btn') and self.new_invoice_btn.winfo_viewable():
                            next_widget = self.new_invoice_btn
                        else:
                            next_widget = self.save_btn
                    elif hasattr(self, 'new_invoice_btn') and next_widget == self.new_invoice_btn and not self.new_invoice_btn.winfo_viewable():
                        next_widget = self.save_btn
                        
                if next_widget and next_widget.winfo_exists():
                    next_widget.focus_set()
                    if hasattr(next_widget, 'select_range'):
                        try:
                            next_widget.select_range(0, tk.END)
                        except:
                            pass
            else:
                first_nav_widget = self.navigation_widgets[0]
                if first_nav_widget and first_nav_widget.winfo_exists():
                    first_nav_widget.focus_set()
                    if hasattr(first_nav_widget, 'select_range'):
                        try:
                            first_nav_widget.select_range(0, tk.END)
                        except:
                            pass
        except Exception as e:
            print(f"Next widget navigasyon hatası: {e}")
        
        return 'break'
    
    def navigate_to_prev_widget(self, current_index):
        """Önceki widget'a geç"""
        if self.suggestions_listbox.winfo_viewable() and self.dialog.focus_get() not in [self.urun_entry, self.suggestions_listbox]:
            self.hide_suggestions()

        try:
            prev_widget_index = current_index - 1
            
            if self.navigation_widgets[current_index] == self.fatura_entry:
                self.tarih_entry.year_entry.focus_set()
                self.tarih_entry.year_entry.select_range(0, tk.END)
                return 'break'
            
            if current_index >=0 and self.navigation_widgets[current_index] == self.tarih_entry:
                self.islem_turu_combo.focus_set()
                self.islem_turu_combo.select_range(0, tk.END)
                return 'break'

            if prev_widget_index >= 0:
                prev_widget = self.navigation_widgets[prev_widget_index]
                
                if not prev_widget.winfo_viewable():
                    if hasattr(self, 'new_invoice_btn') and prev_widget == self.new_invoice_btn and not self.new_invoice_btn.winfo_viewable():
                        if hasattr(self, 'save_continue_btn') and self.save_continue_btn.winfo_viewable():
                            prev_widget = self.save_continue_btn
                        else:
                            prev_widget = self.fiyat_entry 
                    elif hasattr(self, 'save_continue_btn') and prev_widget == self.save_continue_btn and not self.save_continue_btn.winfo_viewable():
                        prev_widget = self.fiyat_entry
                
                if prev_widget and prev_widget.winfo_exists():
                    prev_widget.focus_set()
                    if hasattr(prev_widget, 'select_range'):
                        try:
                            prev_widget.select_range(0, tk.END)
                        except:
                            pass
            else:
                last_nav_widget = None
                for widget in reversed(self.navigation_widgets):
                    if widget.winfo_viewable():
                        last_nav_widget = widget
                        break
                
                if last_nav_widget and last_nav_widget.winfo_exists():
                    last_nav_widget.focus_set()
                    if hasattr(last_nav_widget, 'select_range'):
                        try:
                            last_nav_widget.select_range(0, tk.END)
                        except:
                            pass
        except Exception as e:
            print(f"Prev widget navigasyon hatası: {e}")
        
        return 'break'
    
    def on_urun_enter(self, event):
        """Ürün alanında Enter tuşu"""
        try:
            if self.suggestions_listbox.winfo_viewable() and self.suggestions_listbox.size() > 0:
                selection = self.suggestions_listbox.curselection()
                if selection:
                    self.confirm_suggestion_selection()
                else:
                    self.suggestions_listbox.selection_set(0)
                    self.confirm_suggestion_selection()
                return 'break'

            self.miktar_entry.focus_set()
            self.miktar_entry.select_range(0, tk.END)
            return 'break'
        except Exception as e:
            print(f"Ürün enter hatası: {e}")
            return 'break'

    def on_urun_down(self, event):
        """Ürün alanında Aşağı ok tuşu"""
        if self.suggestions_listbox.winfo_viewable() and self.suggestions_listbox.size() > 0:
            self.suggestions_listbox.focus_set()
            self.suggestions_listbox.selection_set(0)
            return 'break'
        return None
    
    def _on_listbox_selection_changed(self, event=None):
        """Listbox seçim değişimi"""
        pass
    
    def on_suggestion_enter(self, event=None):
        """Öneri listesinde Enter"""
        self.confirm_suggestion_selection()
        return 'break'
    
    def on_suggestion_click(self, event):
        """Öneri listesinde mouse tıklaması"""
        self.confirm_suggestion_selection()
        return 'break'
    
    def on_suggestion_tab(self, event=None):
        """Öneri listesinde Tab"""
        self.confirm_suggestion_selection()
        self.miktar_entry.focus_set()
        self.miktar_entry.select_range(0, tk.END)
        return 'break'
    
    def on_suggestion_down(self, event=None):
        """Öneri listesinde aşağı ok"""
        return None
    
    def on_suggestion_up(self, event=None):
        """Öneri listesinde yukarı ok"""
        current_selection = self.suggestions_listbox.curselection()
        if not current_selection or current_selection[0] == 0:
            self.urun_entry.focus_set()
            self.urun_entry.select_range(0, tk.END)
            return 'break'
        return None
    
    def confirm_suggestion_selection(self):
        """Seçilen öneriyi onayla"""
        selection = self.suggestions_listbox.curselection()
        if selection:
            text = self.suggestions_listbox.get(selection[0])
            if "Ürün bulunamadı" not in text:
                urun_ismi = text.split(' (')[0]
                self.urun_isim_var.set(urun_ismi)
                self.hide_suggestions()
                self.update_stock_display()
                self.miktar_entry.focus_set()
                self.miktar_entry.select_range(0, tk.END)
            else:
                self.hide_suggestions()
                self.urun_entry.focus_set()
                self.urun_entry.select_range(0, tk.END)

    def create_stock_display(self, parent):
        """Stok gösterim bölümü"""
        self.stock_frame = ttk.LabelFrame(parent, text="Fatura Tarihi Stok Durumu", padding=10)
        self.stock_frame.grid(row=7, column=0, columnspan=2, sticky="ew", pady=10)
        
        self.stock_label = ttk.Label(self.stock_frame, text="Ürün seçin ve tarih girin", 
                                    font=('Arial', 10, 'bold'))
        self.stock_label.pack()
    
    def load_data(self):
        """Verileri yükle"""
        if self.islem_id:
            islem = self.stok_manager.get_islem(self.islem_id)
            if islem:
                self.islem_turu_var.set(islem['islemTuru'])
                self.tarih_entry.set_date(islem['tarih'])
                self.fatura_no_var.set(islem['faturaNo'])
                self.urun_isim_var.set(islem['urunIsim'])
                self.miktar_var.set(str(islem['miktar']))
                self.fiyat_var.set(str(islem['fiyat']))
                self.update_stock_display()
    
    def on_islem_turu_changed(self, event=None):
        """İşlem türü değiştiğinde"""
        self.update_stock_display()
    
    def on_tarih_changed(self, *args):
        """Tarih değiştiğinde"""
        self.update_stock_display()
    
    def on_urun_search(self, event=None):
        """Ürün arama"""
        if self.dialog.focus_get() == self.urun_entry or event:
            query = self.urun_isim_var.get().strip()
            if len(query) >= 1:
                suggestions = self.stok_manager.search_stok_kartlari(query)
                self.show_suggestions(suggestions)
                self.update_stock_display()
            else:
                self.hide_suggestions()
        else:
            self.hide_suggestions()
    
    def on_urun_focus(self, event=None):
        """Ürün alanına focus geldiğinde"""
        self.on_urun_search()
    
    def show_suggestions(self, suggestions):
        """Önerileri göster"""
        self.suggestions_listbox.delete(0, tk.END)
        
        if suggestions:
            for stok in suggestions[:10]:
                try:
                    tarih_iso = self.tarih_entry.get_iso_date()
                    mevcut_stok = self.stok_manager.hesapla_stok_tarihe_kadar(stok['urunIsmi'], tarih_iso)
                except Exception as e:
                    mevcut_stok = 0
                
                display_text = f"{stok['urunIsmi']} ({stok['kdo']}) - Stok: {self.stok_manager.format_sayi_basit(mevcut_stok)} {stok['birim']}"
                self.suggestions_listbox.insert(tk.END, display_text)
            
            self.suggestions_listbox.pack(fill=tk.BOTH, expand=True)
        else:
            self.suggestions_listbox.insert(tk.END, "Ürün bulunamadı")
            self.suggestions_listbox.pack(fill=tk.BOTH, expand=True)
    
    def hide_suggestions(self, event=None):
        """Önerileri gizle"""
        if self.suggestions_listbox.winfo_viewable():
            self.suggestions_listbox.pack_forget()
    
    def update_stock_display(self):
        """Stok gösterimini güncelle"""
        urun_ismi = self.urun_isim_var.get().strip()
        
        if urun_ismi:
            try:
                tarih_iso = self.tarih_entry.get_iso_date()
                mevcut_stok = self.stok_manager.hesapla_stok_tarihe_kadar(urun_ismi, tarih_iso)
                stok_karti = self.stok_manager.get_stok_karti_by_name(urun_ismi)
                birim = stok_karti['birim'] if stok_karti else 'ADET'
                
                if mevcut_stok > 0:
                    color = "green"
                    icon = "✅"
                elif mevcut_stok < 0:
                    color = "red"
                    icon = "❌"
                else:
                    color = "gray"
                    icon = "⚪"
                
                try:
                    tarih_formatted = self.tarih_entry.get_date()
                except:
                    tarih_formatted = "Geçersiz tarih"
                
                text = f"{icon} {self.stok_manager.format_sayi_basit(mevcut_stok)} {birim} ({tarih_formatted} stoku)"
                
                self.stock_label.config(text=text, foreground=color)
            except ValueError:
                self.stock_label.config(text="Geçersiz tarih formatı", foreground="red")
            except Exception as e:
                print(f"Stok görüntüleme hatası: {e}")
                self.stock_label.config(text="Ürün veya tarih hatası", foreground="red")
        else:
            self.stock_label.config(text="Ürün seçin ve tarih girin", foreground="black")
    
    def save_and_continue(self):
        """Kaydet ve devam et"""
        if self.save_form(close_dialog=False):
            self.continuous_entry = True
            self.last_form_data = {
                'islem_turu': self.islem_turu_var.get(),
                'tarih': self.tarih_entry.get_iso_date(),
                'fatura_no': self.fatura_no_var.get()
            }
            
            self.urun_isim_var.set("")
            self.miktar_var.set("")
            self.fiyat_var.set("")
            self.hide_suggestions()
            self.update_stock_display()
            
            if hasattr(self, 'save_continue_btn') and self.save_continue_btn.winfo_exists():
                self.save_continue_btn.pack(side=tk.LEFT, padx=5)
            
            if hasattr(self, 'new_invoice_btn') and self.new_invoice_btn.winfo_exists():
                self.new_invoice_btn.pack(side=tk.LEFT, padx=5)
            
            self.dialog.title("Çoklu Ürün Girişi - Aynı Fatura")
            
            self.urun_entry.focus_set()
    
    def start_new_invoice(self):
        """Yeni fatura başlat"""
        self.fatura_no_var.set("")
        self.urun_isim_var.set("")
        self.miktar_var.set("")
        self.fiyat_var.set("")
        self.hide_suggestions()
        self.update_stock_display()
        
        if hasattr(self, 'new_invoice_btn') and self.new_invoice_btn.winfo_exists():
            self.new_invoice_btn.pack_forget()
        if hasattr(self, 'save_continue_btn') and self.save_continue_btn.winfo_exists():
            self.save_continue_btn.pack(side=tk.LEFT, padx=5)
        
        self.dialog.title("Yeni Fatura - Çoklu Ürün Girişi")
        
        self.fatura_entry.focus_set()
        self.fatura_entry.select_range(0, tk.END)
        messagebox.showinfo("Bilgi", "Yeni fatura için fatura numarasını girin!")
    
    def save_form(self, close_dialog=True):
        """Form'u kaydet"""
        try:
            fatura_no = self.fatura_no_var.get().strip()
            urun_isim = self.urun_isim_var.get().strip()
            miktar_str = self.miktar_var.get().strip()
            fiyat_str = self.fiyat_var.get().strip()

            if not all([fatura_no, urun_isim, miktar_str, fiyat_str]):
                messagebox.showerror("Hata", "Lütfen tüm alanları doldurun!")
                return False
            
            try:
                # Türkçe virgül formatını noktaya çevir
                miktar_str_clean = miktar_str.replace(',', '.')
                fiyat_str_clean = fiyat_str.replace(',', '.')
                
                miktar = float(miktar_str_clean)
                fiyat = float(fiyat_str_clean)
                tarih_iso = self.tarih_entry.get_iso_date()
            except ValueError as e:
                error_msg = str(e).lower()
                if "tarih" in error_msg or "geçersiz" in error_msg:
                    messagebox.showerror("Hata", "Geçerli bir tarih girin!")
                else:
                    messagebox.showerror("Hata", "Miktar ve fiyat sayı olmalıdır!")
                return False
            except Exception as e:
                print(f"Form validation hatası: {e}")
                messagebox.showerror("Hata", "Form doğrulama hatası!")
                return False
            
            if miktar <= 0 or fiyat <= 0:
                messagebox.showerror("Hata", "Miktar ve fiyat sıfırdan büyük olmalıdır!")
                return False
            
            islem_data = {
                'islemTuru': self.islem_turu_var.get(),
                'tarih': tarih_iso,
                'faturaNo': fatura_no,
                'urunIsim': urun_isim,
                'miktar': miktar,
                'fiyat': fiyat,
                'birim': 'ADET'
            }
            
            stok_karti = self.stok_manager.get_stok_karti_by_name(islem_data['urunIsim'])
            if stok_karti:
                islem_data['birim'] = stok_karti['birim']
            
            if self.islem_id:
                self.stok_manager.update_islem(self.islem_id, islem_data)
                self.stok_manager.save_data()
                messagebox.showinfo("Başarılı", "İşlem güncellendi!")
                if close_dialog:
                    self.ok()
                return True
            else:
                self.stok_manager.add_islem(islem_data)
                self.stok_manager.save_data()
                
                if not self.continuous_entry and close_dialog:
                    if islem_data['islemTuru'] == 'GİRİŞ':
                        messagebox.showinfo("Başarılı", f"✅ Giriş işlemi başarılı!\n\n{self.stok_manager.format_sayi_basit(miktar)} {islem_data['birim']} giriş yapıldı.")
                    else:
                        kalan_stok = self.stok_manager.hesapla_stok_tarihe_kadar(islem_data['urunIsim'], islem_data['tarih'])
                        messagebox.showinfo("Başarılı", f"✅ Çıkış işlemi başarılı!\n\n{self.stok_manager.format_sayi_basit(miktar)} {islem_data['birim']} çıkış yapıldı.\nKalan stok: {self.stok_manager.format_sayi_basit(kalan_stok)} {islem_data['birim']}")
                    self.ok()
                
                return True
            
        except Exception as e:
            print(f"Kaydetme hatası detayı: {e}")
            messagebox.showerror("Hata", f"Kaydetme hatası: {e}")
            return False
    
    def save(self):
        """Normal kaydet"""
        self.continuous_entry = False
        self.save_form(close_dialog=True)

class StokKartiDialog(BaseDialog):
    """Stok kartı ekleme/düzenleme dialog'u"""
    def __init__(self, parent, stok_manager, kdo=None):
        self.stok_manager = stok_manager
        self.kdo = kdo
        
        title = "Stok Kartını Düzenle" if kdo else "Yeni Stok Kartı Ekle"
        super().__init__(parent, title, "500x300")
        
        self.create_widgets()
        self.load_data()
        
        self.dialog.wait_window()
    
    def create_widgets(self):
        """Widget'ları oluştur"""
        main_frame = ttk.Frame(self.dialog)
        main_frame.pack(fill=tk.BOTH, expand=True, padx=20, pady=20)
        
        info_label = ttk.Label(main_frame, text="💡 Stok kodu otomatik üretilir, isterseniz değiştirebilirsiniz", 
                               foreground="blue")
        info_label.grid(row=0, column=0, columnspan=2, pady=(0, 15))
        
        ttk.Label(main_frame, text="Stok Kodu:").grid(row=1, column=0, sticky="w", pady=5)
        self.kdo_var = tk.StringVar()
        self.kdo_entry = ttk.Entry(main_frame, textvariable=self.kdo_var, width=40)
        self.kdo_entry.grid(row=1, column=1, sticky="ew", pady=5)
        
        ttk.Label(main_frame, text="Ürün İsmi:").grid(row=2, column=0, sticky="w", pady=5)
        self.urun_ismi_var = tk.StringVar()
        ttk.Entry(main_frame, textvariable=self.urun_ismi_var, width=40).grid(row=2, column=1, sticky="ew", pady=5)
        
        ttk.Label(main_frame, text="Birim:").grid(row=3, column=0, sticky="w", pady=5)
        self.birim_var = tk.StringVar(value="ADET")
        birim_combo = ttk.Combobox(main_frame, textvariable=self.birim_var, 
                                  values=["ADET", "KG", "M", "PAKET"], state="readonly", width=37)
        birim_combo.grid(row=3, column=1, sticky="ew", pady=5)
        
        button_frame = ttk.Frame(main_frame)
        button_frame.grid(row=4, column=0, columnspan=2, pady=20)
        
        ttk.Button(button_frame, text="İptal", command=self.cancel).pack(side=tk.LEFT, padx=5)
        ttk.Button(button_frame, text="Kaydet", command=self.save).pack(side=tk.LEFT, padx=5)
        
        main_frame.columnconfigure(1, weight=1)
    
    def load_data(self):
        """Verileri yükle"""
        if self.kdo:
            stok_karti = self.stok_manager.get_stok_karti_by_kdo(self.kdo)
            if stok_karti:
                self.kdo_var.set(stok_karti['kdo'])
                self.urun_ismi_var.set(stok_karti['urunIsmi'])
                self.birim_var.set(stok_karti['birim'])
        else:
            auto_kdo = self.stok_manager.generate_auto_stok_kodu()
            self.kdo_var.set(auto_kdo)
    
    def save(self):
        """Kaydet"""
        try:
            kdo = self.kdo_var.get().strip()
            urun_ismi = self.urun_ismi_var.get().strip()
            birim = self.birim_var.get()
            
            if not kdo or not urun_ismi:
                messagebox.showerror("Hata", "Lütfen tüm alanları doldurun!")
                return
            
            if self.kdo:
                self.stok_manager.update_stok_karti(self.kdo, kdo, urun_ismi, birim)
                messagebox.showinfo("Başarılı", "Stok kartı güncellendi!")
            else:
                self.stok_manager.add_stok_karti(kdo, urun_ismi, birim)
                messagebox.showinfo("Başarılı", "Stok kartı eklendi!")
            
            self.stok_manager.save_data()
            self.ok()
            
        except Exception as e:
            messagebox.showerror("Hata", f"Kaydetme hatası: {e}")

class UrunDetayDialog(BaseDialog):
    """Ürün detay raporu dialog'u"""
    def __init__(self, parent, stok_manager, urun_ismi):
        self.stok_manager = stok_manager
        self.urun_ismi = urun_ismi
        
        super().__init__(parent, f"{urun_ismi} - Muavin Dökümü", "900x600")
        
        self.create_widgets()
        self.load_data()
        
        self.dialog.wait_window()
    
    def create_widgets(self):
        """Widget'ları oluştur"""
        main_frame = ttk.Frame(self.dialog)
        main_frame.pack(fill=tk.BOTH, expand=True, padx=20, pady=20)
        
        tree_frame = ttk.Frame(main_frame)
        tree_frame.pack(fill=tk.BOTH, expand=True)
        
        tree_scroll = ttk.Scrollbar(tree_frame)
        tree_scroll.pack(side=tk.RIGHT, fill=tk.Y)
        
        columns = ("Tarih", "İşlem", "Fatura No", "Miktar", "Fiyat", "Tutar", "Kalan Stok")
        self.tree = ttk.Treeview(tree_frame, columns=columns, show='headings', 
                                yscrollcommand=tree_scroll.set)
        tree_scroll.config(command=self.tree.yview)
        
        for col in columns:
            self.tree.heading(col, text=col)
            self.tree.column(col, width=100, anchor='center')
        
        self.tree.column("Fatura No", width=120)
        self.tree.column("Tarih", width=90)
        
        self.tree.pack(fill=tk.BOTH, expand=True)
        
        self.create_summary_frame(main_frame)
        
        button_frame = ttk.Frame(main_frame)
        button_frame.pack(fill=tk.X, pady=10)
        
        ttk.Button(button_frame, text="Kapat", command=self.cancel).pack(side=tk.LEFT)
        ttk.Button(button_frame, text="⬇️ Excel'e Aktar", command=self.export_excel).pack(side=tk.LEFT, padx=10)
    
    def create_summary_frame(self, parent):
        """Özet bilgiler frame'i"""
        self.summary_frame = ttk.LabelFrame(parent, text="Özet Bilgiler", padding=10)
        self.summary_frame.pack(fill=tk.X, pady=10)
        
        self.summary_vars = {
            'toplam_giris': tk.StringVar(),
            'toplam_cikis': tk.StringVar(),
            'kalan_stok': tk.StringVar(),
            'giris_tutari': tk.StringVar(),
            'cikis_tutari': tk.StringVar(),
            'islem_sayisi': tk.StringVar()
        }
        
        labels = [
            ("Toplam Giriş:", 'toplam_giris'),
            ("Toplam Çıkış:", 'toplam_cikis'),
            ("Kalan Stok:", 'kalan_stok'),
            ("Giriş Tutarı:", 'giris_tutari'),
            ("Çıkış Tutarı:", 'cikis_tutari'),
            ("İşlem Sayısı:", 'islem_sayisi')
        ]
        
        for i, (label_text, var_key) in enumerate(labels):
            row = i // 3
            col = (i % 3) * 2
            
            ttk.Label(self.summary_frame, text=label_text, font=('Arial', 10, 'bold')).grid(
                row=row, column=col, sticky="w", padx=5, pady=2)
            ttk.Label(self.summary_frame, textvariable=self.summary_vars[var_key]).grid(
                row=row, column=col+1, sticky="w", padx=5, pady=2)
    
    def load_data(self):
        """Verileri yükle"""
        rapor = self.stok_manager.get_urun_detay_raporu(self.urun_ismi)
        
        for i in self.tree.get_children():
            self.tree.delete(i)

        for detay in rapor['detaylar']:
            islem_icon = "🟢" if detay['islemTuru'] == 'GİRİŞ' else "🔴"
            miktar_text = f"{'+'if detay['islemTuru'] == 'GİRİŞ' else '-'}{self.stok_manager.format_sayi_basit(detay['miktar'])} {detay.get('birim', 'ADET')}"
            
            kalan_stok = detay['kalanStok']
            kalan_text = f"{self.stok_manager.format_sayi_basit(kalan_stok)} {detay.get('birim', 'ADET')}"
            
            tarih_formatted = self.stok_manager.format_tarih(detay['tarih'])
            
            values = (
                tarih_formatted,
                f"{islem_icon} {detay['islemTuru']}",
                detay['faturaNo'],
                miktar_text,
                self.stok_manager.format_sayi(detay['fiyat']),
                self.stok_manager.format_sayi(detay['tutar']),
                kalan_text
            )
            
            self.tree.insert("", tk.END, values=values)
        
        ozet = rapor['ozet']
        birim = rapor['detaylar'][0].get('birim', 'ADET') if rapor['detaylar'] else 'ADET'
        
        self.summary_vars['toplam_giris'].set(f"{self.stok_manager.format_sayi_basit(ozet['toplamGiris'])} {birim}")
        self.summary_vars['toplam_cikis'].set(f"{self.stok_manager.format_sayi_basit(ozet['toplamCikis'])} {birim}")
        self.summary_vars['kalan_stok'].set(f"{self.stok_manager.format_sayi_basit(ozet['kalanStok'])} {birim}")
        self.summary_vars['giris_tutari'].set(self.stok_manager.format_sayi(ozet['toplamGirisTutar']))
        self.summary_vars['cikis_tutari'].set(self.stok_manager.format_sayi(ozet['toplamCikisTutar']))
        self.summary_vars['islem_sayisi'].set(str(ozet['islemSayisi']))
    
    def export_excel(self):
        """Excel'e aktar"""
        try:
            from .reports import ReportExporter
            
            safe_name = "".join(c for c in self.urun_ismi if c.isalnum() or c in (' ', '-', '_')).rstrip()
            filename = filedialog.asksaveasfilename(
                defaultextension=".csv",
                filetypes=[("CSV files", "*.csv")],
                title="Ürün Detay Raporu Excel'e Aktar",
                initialfile=f"{safe_name}_detay_raporu.csv"
            )
            
            if filename:
                exporter = ReportExporter(self.stok_manager)
                exporter.export_urun_detay_csv(filename, self.urun_ismi)
                messagebox.showinfo("Başarılı", "Ürün detay raporu Excel dosyası oluşturuldu!")
        except Exception as e:
            messagebox.showerror("Hata", f"Excel dışa aktarma hatası: {e}")

class FifoRaporDialog(BaseDialog):
    """FIFO maliyet raporu dialog'u"""
    def __init__(self, parent, stok_manager):
        self.stok_manager = stok_manager
        
        super().__init__(parent, "FIFO Maliyet Raporu", "700x500")
        
        self.create_widgets()
        self.load_data()
        
        self.dialog.wait_window()
    
    def create_widgets(self):
        """Widget'ları oluştur"""
        main_frame = ttk.Frame(self.dialog)
        main_frame.pack(fill=tk.BOTH, expand=True, padx=20, pady=20)
        
        tree_frame = ttk.Frame(main_frame)
        tree_frame.pack(fill=tk.BOTH, expand=True)
        
        tree_scroll = ttk.Scrollbar(tree_frame)
        tree_scroll.pack(side=tk.RIGHT, fill=tk.Y)
        
        columns = ("Ürün İsmi", "Kalan Miktar (FIFO)", "Alış Maliyeti Değeri (FIFO)")
        self.tree = ttk.Treeview(tree_frame, columns=columns, show='headings', 
                                yscrollcommand=tree_scroll.set)
        tree_scroll.config(command=self.tree.yview)
        
        self.tree.heading("Ürün İsmi", text="Ürün İsmi")
        self.tree.heading("Kalan Miktar (FIFO)", text="Kalan Miktar (FIFO)")
        self.tree.heading("Alış Maliyeti Değeri (FIFO)", text="Alış Maliyeti Değeri (FIFO)")
        
        self.tree.column("Ürün İsmi", width=300, anchor='w')
        self.tree.column("Kalan Miktar (FIFO)", width=150, anchor='center')
        self.tree.column("Alış Maliyeti Değeri (FIFO)", width=200, anchor='e')
        
        self.tree.pack(fill=tk.BOTH, expand=True)
        
        self.total_frame = ttk.LabelFrame(main_frame, text="Genel Toplam", padding=10)
        self.total_frame.pack(fill=tk.X, pady=10)
        
        self.total_label = ttk.Label(self.total_frame, text="", font=('Arial', 12, 'bold'))
        self.total_label.pack()
        
        button_frame = ttk.Frame(main_frame)
        button_frame.pack(fill=tk.X, pady=10)
        
        ttk.Button(button_frame, text="Kapat", command=self.cancel).pack(side=tk.LEFT)
        ttk.Button(button_frame, text="⬇️ Excel'e Aktar", command=self.export_excel).pack(side=tk.LEFT, padx=10)
    
    def load_data(self):
        """Verileri yükle"""
        for i in self.tree.get_children():
            self.tree.delete(i)

        fifo_data = self.stok_manager.calculate_fifo_report()
        total_value = 0.0
        
        if fifo_data:
            for item in fifo_data:
                total_value += item['alisMaliyetiDegeri']
                
                values = (
                    item['urunIsmi'],
                    f"{self.stok_manager.format_sayi_basit(item['kalanMiktarFIFO'])} {item['birim']}",
                    f"{self.stok_manager.format_sayi(item['alisMaliyetiDegeri'])} TL"
                )
                
                self.tree.insert("", tk.END, values=values)
        else:
            self.tree.insert("", tk.END, values=("Henüz FIFO raporu için veri yok", "", ""))
        
        self.total_label.config(text=f"Toplam FIFO Maliyet Değeri: {self.stok_manager.format_sayi(total_value)} TL")
    
    def export_excel(self):
        """Excel'e aktar"""
        try:
            from .reports import ReportExporter
            
            filename = filedialog.asksaveasfilename(
                defaultextension=".csv",
                filetypes=[("CSV files", "*.csv")],
                title="FIFO Maliyet Raporu Excel'e Aktar",
                initialfile="fifo_maliyet_raporu.csv"
            )
            
            if filename:
                exporter = ReportExporter(self.stok_manager)
                fifo_data = self.stok_manager.calculate_fifo_report()
                exporter.export_fifo_report_csv(filename, fifo_data)
                messagebox.showinfo("Başarılı", "FIFO maliyet raporu Excel dosyası oluşturuldu!")
        except Exception as e:
            messagebox.showerror("Hata", f"Excel dışa aktarma hatası: {e}")